//=============================================================================
// VisuStella MZ - Message Core
// VisuMZ_1_MessageCore.js
//=============================================================================

var Imported = Imported || {};
Imported.VisuMZ_1_MessageCore = true;

var VisuMZ = VisuMZ || {};
VisuMZ.MessageCore = VisuMZ.MessageCore || {};
VisuMZ.MessageCore.version = 1.27;

//=============================================================================
 /*:
 * @target MZ
 * @plugindesc [RPG Maker MZ] [Tier 1] [Version 1.27] [MessageCore]
 * @author VisuStella
 * @url http://www.yanfly.moe/wiki/Message_Core_VisuStella_MZ
 * @orderAfter VisuMZ_0_CoreEngine
 *
 * @help
 * ============================================================================
 * Introduction
 * ============================================================================
 *
 * The Message Core plugin extends and builds upon the message functionality of
 * RPG Maker MZ and allows you, the game dev, to customize the workflow for
 * your game's message system.
 *
 * Features include all (but not limited to) the following:
 *
 * * Control over general message settings.
 * * Auto-Color key words and/or database entries.
 * * Increases the text codes available to perform newer functions/effects.
 * * Ability for you to implement custom Text Code actions.
 * * Ability for you to implement custom Text code string replacements.
 * * Invoke a macro system to speed up the dev process.
 * * Add a Text Speed option to the Options menu.
 * * Add the ever so useful Word Wrap to your message system.
 * * Extend the choice selection process to your liking.
 * * The ability to enable/disable as well as show/hide certain choices.
 *
 * ============================================================================
 * Requirements
 * ============================================================================
 *
 * This plugin is made for RPG Maker MZ. This will not work in other iterations
 * of RPG Maker.
 *
 * ------ Tier 1 ------
 *
 * This plugin is a Tier 1 plugin. Place it under other plugins of lower tier
 * value on your Plugin Manager list (ie: 0, 1, 2, 3, 4, 5). This is to ensure
 * that your plugins will have the best compatibility with the rest of the
 * VisuStella MZ library.
 *
 * ============================================================================
 * Major Changes
 * ============================================================================
 *
 * This plugin adds some new hard-coded features to RPG Maker MZ's functions.
 * The following is a list of them.
 *
 * ---
 * 
 * Dim Background Extension
 * 
 * Before, when using the Dim Background as a part of a Show Text event, its
 * size is only the same as the message window's width itself. This looked
 * really ugly because it had hard edges cutting off while gradients are seen
 * elsewhere. To make it look better, we extended the dimmed background to span
 * the width of the screen instead.
 * 
 * ---
 * 
 * Extended Messages
 * 
 * If you decide to expand the size of the message window to allow for more
 * rows to be displayed, you can type in the data for them by chaining together
 * Show Message events. They will take data from each other and display them in
 * the same message window as long as there are enough rows.
 * 
 * ---
 *
 * Extended Choice Lists
 * 
 * Choice lists can be extended by just chaining one Choice List event after
 * the other in succession along the same indentation. They do not extend if
 * there is any event other than a Choice List option between them on the same
 * indentation level.
 *
 * ---
 *
 * ============================================================================
 * Available Text Codes
 * ============================================================================
 *
 * The following are text codes that you may use with this plugin. Some of
 * these are original text codes provided by RPG Maker MZ, while others are
 * new text codes added through this plugin. You may even add your own text
 * codes through the plugin parameters.
 *
 * === RPG Maker MZ Text Codes ===
 *
 * The following are text codes that come with RPG Maker MZ. These text codes
 * cannot be edited through the Plugin Parameters.
 *
 * ---
 *
 * ------------------   -------------------------------------------------------
 * Text Code            Effect (Global)
 * ------------------   -------------------------------------------------------
 * \V[x]                Replaced by the value of variable 'x'.
 * \N[x]                Replaced by the name of actor 'x'.
 * \P[x]                Replaced by the name of party member 'x'.
 * \C[x]                Draw the subsequent text with window skin color 'x'.
 * \I[x]                Draw icon 'x'.
 *
 * \PX[x]               Moves text x position to 'x'.
 * \PY[x]               Moves text y position to 'y'.
 *
 * \G                   Replaced by the currency unit.
 *
 * \{                   Increase the text font size by one step.
 * \}                   Decrease the text font size by one step.
 * \FS[x]               Changes the text font size to 'x'.
 *
 * \\                   Replaced by the backslash character.
 *
 * ---
 *
 * ------------------   -------------------------------------------------------
 * Text Code            Effect (Message Window Only)
 * ------------------   -------------------------------------------------------
 * \$                   Opens the gold window.
 * \.                   Waits a 1/4 second.
 * \|                   Waits a full second.
 * \!                   Waits for button input.
 * \>                   Display remaining text on same line all at once.
 * \<                   Cancel the effect that displays text all at once.
 * \^                   Do not wait for input after displaying text to move on.
 *
 * ---
 *
 * === Message Core Hard-Coded Text Codes ===
 *
 * The following text codes are hard-coded into VisuStella MZ Message Core's
 * code. These text codes cannot be edited through the Plugin Parameters.
 * 
 * ---
 *
 * ------------------   -------------------------------------------------------
 * Text Code            Effect (Global)
 * ------------------   -------------------------------------------------------
 * <b>                  Makes subsequent text bold.
 * </b>                 Removes bold from subsequent text.
 * <i>                  Makes subsequent text italic.
 * </i>                 Removes italic from subsequent text.
 * 
 * <left>               Makes subsequent text left-aligned.
 * </left>              Removes left-alignment for subsequent text.
 * <center>             Makes subsequent text center-aligned.
 * </center>            Removes center-alignment for subsequent text.
 * <right>              Makes subsequent text right-aligned.
 * </right>             Removes right-alignment for subsequent text.
 *
 * Note1: Use at line-start.
 *
 * <ColorLock>          Text codes can't change text color for subsequent text.
 * </ColorLock>         Removes Color Lock property.
 *
 * <WordWrap>           Enables Word Wrap for this window. *Note2*
 * </WordWrap>          Disables Word Wrap for this window. *Note2*
 * <br>                 Adds a line break. Requires Word Wrap enabled.
 * <line break>         Adds a line break. Requires Word Wrap enabled.
 *
 * Note2: Some windows cannot use Word Wrap such as the Choice Window.
 *
 * \picture<x>          Draws picture x (filename) at current text position.
 * \CenterPicture<x>    Draws picture x (filename) centered at the window.
 *
 * ---
 *
 * ------------------   -------------------------------------------------------
 * Text Code            Effect (Message Window Only)
 * ------------------   -------------------------------------------------------
 * \CommonEvent[x]      Runs common event x when text code is reached.
 * \Wait[x]             Makes the message wait x frames before continuing.
 * 
 * <Auto>               Resizes message window dimensions to fit text. *Note3*
 * <Auto Width>         Resizes message window width to fit text. *Note3*
 * <Auto Height>        Resizes message window height to fit text. *Note3*
 * 
 * <Auto Actor: x>      Resizes message window and positions it over actor x
 *                      sprite's head. *Note3*
 * <Auto Party: x>      Resizes message window and positions it over party
 *                      member x sprite's head. *Note3*
 * <Auto Player>        Map-Only. Resizes message window and positions it over
 *                      the player sprite's head. *Note3*
 * <Auto Event: x>      Map-Only. Resizes message window and positions it over
 *                      event x sprite's head. *Note3*
 * <Auto Enemy: x>      Battle-Only. Resizes message window and positions it
 *                      over enemy x sprite's head. *Note3*
 *
 * Note3: Upon using these text codes, the message window's settings will be
 * reset for the upcoming message. These effects do not work with Word Wrap.
 *
 * ---
 *
 * ----------------------------   ---------------------------------------------
 * Text Code                      Effect (Battle Only)
 * ----------------------------   ---------------------------------------------
 * <Current Battle Target>        Replaces text code with the current target of
 *                                an action in battle.
 * <Current Battle User>          Replaces text code with the currently active
 *                                user in battle.
 * <Current Battle Action>        Replaces text code with the current battle
 *                                action's name with an icon in front.
 * <Current Battle Action Name>   Replaces text code with the current battle
 *                                action's name without an icon.
 * 
 * If there is no battle, no target, no user, or no action, then the text code
 * will just be replaced with no text.
 * 
 * These text codes are NOT recommended to be used inside of Help Descriptions.
 * They are best used with "Show Text" event commands.
 *
 * ---
 *
 * -----------------------------  ---------------------------------------------
 * Text Code                      Effect (Choice Window Only)
 * -----------------------------  ---------------------------------------------
 * <Show>                         Choice is always shown.
 * <Show Switch: x>               Choice shown if switch x is ON.
 * <Show Switches: x,x,x>         Choice shown if the x switches are all ON.
 * <Show All Switches: x,x,x>     Choice shown if the x switches are all ON.
 * <Show Any Switches: x,x,x>     Choice shown if any of x switches are ON.
 *
 * <Hide>                         Choice is always hidden.
 * <Hide Switch: x>               Choice hidden if switch x is ON.
 * <Hide Switches: x,x,x>         Choice hidden if the x switches are all ON.
 * <Hide All Switches: x,x,x>     Choice hidden if the x switches are all ON.
 * <Hide Any Switches: x,x,x>     Choice hidden if any of x switches are ON.
 *
 * <Enable>                       Choice is always enabled.
 * <Enable Switch: x>             Choice enabled if switch x is ON.
 * <Enable Switches: x,x,x>       Choice enabled if the x switches are all ON.
 * <Enable All Switches: x,x,x>   Choice enabled if the x switches are all ON.
 * <Enable Any Switches: x,x,x>   Choice enabled if any of x switches are ON.
 *
 * <Disable>                      Choice is always disabled.
 * <Disable Switch: x>            Choice disabled if switch x is ON.
 * <Disable Switches: x,x,x>      Choice disabled if the x switches are all ON.
 * <Disable All Switches: x,x,x>  Choice disabled if the x switches are all ON.
 * <Disable Any Switches: x,x,x>  Choice disabled if any of x switches are ON.
 *
 * ---
 *
 * -----------------  ---------------------------------------------------------
 * Text Code          Effect (Name Window Only)
 * -----------------  ---------------------------------------------------------
 * <Left>             Positions the name box window to the left.
 * <Center>           Positions the name box window to the center.
 * <Right>            Positions the name box window to the right.
 * <Position: x>      Replace 'x' with a number from 0 to 10. This positions
 *                    the name box window on the screen relative to the
 *                    position of the value 'x' represents.
 * \NormalBG          Changes background type of window to normal type.
 * \DimBG             Changes background type of window to dim type.
 * \TransparentBG     Changes background type of window to transparent type.
 *
 * ---
 * 
 * -------------------------------   ------------------------------------------
 * Text Code                         Effect (Message Window Only)
 * -------------------------------   ------------------------------------------
 * 
 * <Position: x, y, width, height>   Forces the message window to exact listed
 *                                   coordinates and dimensions. Replace each
 *                                   of the arguments with numbers. *Note*
 * 
 * <Coordinates: x, y>               Forces the message window to the exact
 *                                   listed coordinates. Replace each of the
 *                                   arguments with numbers. *Note*
 * 
 * <Dimensions: width, height>       Forces the message window size to the
 *                                   exact listed dimensions. Replace each of
 *                                   the arguments with numbers. *Note*
 * 
 * *NOTE* These text codes do not work with Word Wrap.
 * 
 * ---
 *
 * === Message Core Customizable Text Codes ===
 *
 * The following text codes can be altered through the Message Core's various
 * Plugin Parameters to adjust replacements and actions.
 *
 * ---
 *
 * ------------------   -------------------------------------------------------
 * Text Code            Effect (Global)
 * ------------------   -------------------------------------------------------
 * \Class[x]            Draws class x's icon (if have) and name.
 * \ClassName[x]        Draws class x's name only.
 *
 * \Skill[x]            Draws skill x's icon (if have) and name.
 * \SkillName[x]        Draws skill x's name only.
 *
 * \Item[x]             Draws item x's icon (if have) and name.
 * \ItemName[x]         Draws item x's name only.
 * \ItemQuantity[x]     Inserts the number of item x's owned by the party.
 *
 * \Weapon[x]           Draws weapon x's icon (if have) and name.
 * \WeaponName[x]       Draws weapon x's name only.
 * \WeaponQuantity[x]   Inserts the number of weapon x's owned by the party.
 *
 * \Armor[x]            Draws armor x's icon (if have) and name.
 * \ArmorName[x]        Draws armor x's name only.
 * \ArmorQuantity[x]    Inserts the number of armor x's owned by the party.
 *
 * \LastGainObj         Draws the icon + name of the last party-gained object.
 * \LastGainObjName     Draws the name of the last party-gained object.
 * \LastGainObjQuantity Inserts the quantity of the last party-gained object.
 *
 * \State[x]            Draws state x's icon (if have) and name.
 * \StateName[x]        Draws state x's name only.
 *
 * \Enemy[x]            Draws enemy x's icon (if have) and name.
 * \EnemyName[x]        Draws enemy x's name only.
 *
 * \Troop[x]            Draws troop x's icon (if have) and name.
 * \TroopName[x]        Draws troop x's name only.
 *
 * \TroopMember[x]      Draws troop member x's icon (if have) and name. *Note1*
 * \TroopNameMember[x]  Draws troop member x's name only. *Note1*
 * 
 * Note1: Only works in battle.
 *
 * \NormalBG            Changes background type of window to normal type.
 * \DimBG               Changes background type of window to dim type.
 * \TransparentBG       Changes background type of window to transparent type.
 *
 * \FontChange<x>       Changes font face to x font name.
 * \ResetFont           Resets font settings.
 *
 * \ResetColor          Resets color settings.
 * \HexColor<x>         Changes text color to x hex color (ie. #123abc).
 * \OutlineColor[x]     Changes outline color to text color x.
 * \OutlineHexColor<x>  Changes outline color to x hex color (ie. #123abc).
 * \OutlineWidth[x]     Changes outline width to x thickness.
 * 
 * \WindowMoveTo<?>     Moves window to exact coordinates. *Note2*
 * \WindowMoveBy<?>     Moves window by relative values. *Note2*
 * \WindowReset         Resets window position to original position.
 *
 * Note2: Replace '?' with the following format:
 *   targetX, targetY, targetWidth, targetHeight, duration, easingType
 *   Only targetX and targetY are required arguments. These will only alter the
 *   window dimensions when the text has arrived at that point. They will not
 *   alter the window preemptively. This is not used as a window positioner.
 *   Use the <Position: x, y, width, height> text code for that.
 *
 * ---
 *
 * ------------------   -------------------------------------------------------
 * Text Code            Effect (Message Window Only)
 * ------------------   -------------------------------------------------------
 * \ActorFace[x]        Inserts actor x's face into the Message Window.
 * \PartyFace[x]        Inserts party member x's face into the Message Window.
 * \ChangeFace<x,y>     Changes message face to x filename, y index.
 * \FaceIndex[x]        Changes message face index to x.
 *
 * \TextDelay[x]        Sets delay in frames between characters to x frames.
 * 
 * ---
 * 
 * As these text codes can be added, removed, and/or altered, their functions
 * may or may not be the same depending on how you've altered them. VisuStella
 * is not responsible for any errors caused by changes made to pre-made text
 * codes nor any new text codes they did not make.
 * 
 * ============================================================================
 * Plugin Commands
 * ============================================================================
 *
 * The following are Plugin Commands that come with this plugin. They can be
 * accessed through the Plugin Command event command.
 *
 * ---
 * 
 * === Message Plugin Commands ===
 * 
 * ---
 *
 * Message: Properties
 *   Change the various properties of the Message Window.
 *
 *   Rows:
 *   - Change the number of Message Window rows.
 *   - Leave at 0 to keep it unchanged.
 *
 *   Width: 
 *   - Change the Message Window width in pixels.
 *   - Leave at 0 to keep it unchanged.
 *
 *   Center:
 *   - Center the window X after changing its width?
 *
 *   Word Wrap:
 *   - Enable or disable Word Wrap for the Message Window?
 *
 * ---
 * 
 * === Choice Plugin Commands ===
 * 
 * ---
 *
 * Choice: Properties
 *   Change the properties found in the Show Choices event command.
 *
 *   Line Height:
 *   - Change the line height for the show choices.
 *   - Leave at 0 to keep this unchanged.
 *
 *   Max Rows:
 *   - Maximum number of choice rows to be displayed.
 *   - Leave at 0 to keep this unchanged.
 *
 *   Max Columns:
 *   - Maximum number of choice columns to be displayed.
 *   - Leave at 0 to keep this unchanged.
 *
 *   Text Alignment:
 *   - Text alignment for Show Choice window.
 *
 * ---
 *
 * ============================================================================
 * Plugin Parameters: General Settings
 * ============================================================================
 *
 * General settings involving the message system. These settings range from
 * adjust how the Message Window looks to more intricate settings like how
 * some of the default text codes work.
 *
 * ---
 *
 * Message Window
 *
 *   Default Rows:
 *   - Default number of rows to display for the Message Window.
 *
 *   Default Width:
 *   - Default Message Window width in pixels.
 *
 *   Fast Forward Key:
 *   - This is the key used for fast forwarding messages.
 *   - WARNING: If this key is the same as the dash button, this will clear out
 *     any held down inputs upon triggering an event  to prevent players from
 *     skipping potentially useful information stored in messages. If you do
 *     not want the input to be cleared, use a different key.
 *
 *   Text Delay:
 *   - How many frames to wait between characters drawn?
 *   - Use 0 for instant.
 * 
 *   Default Outline Width:
 *   - Changes the default outline width to this many pixels thick.
 *
 * ---
 *
 * Name Box Window
 *
 *   Default Color:
 *   - Default color for the Name Box Window's text.
 *
 *   Offset X:
 *   - How much to offset the name box window X by
 *     (as long as it doesn't go offscreen).
 *
 *   Offset Y:
 *   - How much to offset the name box window Y by
 *     (as long as it doesn't go offscreen).
 *
 * ---
 *
 * Choice List Window
 *
 *   Line Height:
 *   - What is the default line height for Show Choices?
 *
 *   Max Rows:
 *   - Maximum number of rows to visibly display?
 *
 *   Max Columns:
 *   - Maximum number of columns to visibly display?
 *
 *   Text Alignment:
 *   - Default alignment for Show Choice window.
 *
 * ---
 *
 * Default Text Codes
 *
 *   Relative \PX \PY:
 *   - Make \PX[x] and \PY[x] adjust relative starting position than
 *     exact coordinates.
 *
 *   \{ Maximum:
 *   - Determine the maximum size that \{ can reach.
 *
 *   \} Minimum:
 *   - Determine the minimum size that \} can reach.
 *
 *   \{ Change \}
 *   - How much does \{ and \} change font size by?
 *
 * ---
 *
 * ============================================================================
 * Plugin Parameters: Auto-Color Settings
 * ============================================================================
 *
 * For certain windows such as the Message Window, Help Window, and Choice
 * Window, Auto-Color is enabled to automatically highlight and color certain
 * database entries, keywords, and just about anything you, the game dev, wants
 * to be automatically colored. This is done to avoid typing out \C[6]Jack\C[0]
 * every time Jack's name is written out as it will be automatically colored in
 * those specific windows.
 *
 * The Plugin Parameters will give you full reign over which database entries
 * and keywords you want to be automatically colored as long as they follow a
 * few rules:
 * 
 * -----------------
 * Auto-Color Rules:
 * -----------------
 *
 * 1. Database names and keywords are case sensitive.
 *    This means if "Potion" is a marked keyword, typing out "potion" will not
 *    prompt the auto-color to highlight "potion". You must add the lowercase
 *    version of the word into the keyword list if you want it to count.
 *
 * 2. Database names and keywords are exact size (for Roman languages)
 *    This means if "Potion" is a marked keyword, typing out "potions" will not
 *    prompt the auto-color to highlight "potions". You must type out all of
 *    the variations of the words you want affected into the keyword list to
 *    prompt the auto-color highlight.
 * 
 *    This does not apply to Japanese, Korean, or Chinese languages.
 *
 * 3. Possessive cases and other language symbols aren't counted.
 *    Symbols such as periods, commas, quotes, parentheses, and similar symbols
 *    do no count towards Rule 2. This means if "Potion" is a marked keyword,
 *    the typing out "(Potion)" will still highlight the "Potion" part of the
 *    word according to the auto-color.
 * 
 * 4. Names with special characters like !, ?, [, ], etc. will be ignored.
 *    These cause conflicts with how auto-colors are detected.
 *
 * ---
 *
 * Database Highlighting
 *
 *   Actors:
 *   Classes:
 *   Skills:
 *   Items:
 *   Weapons:
 *   Armors:
 *   Enemies:
 *   States:
 *   - Any usage of a the selected database entry's name is auto-colored with
 *     the text code number.
 *   - Use 0 to not auto-color.
 *
 * ---
 *
 * Word Highlighting
 *
 *   \C[x]: Color
 *   - These are lists of all the words that will be automatically colored with
 *     the x text color.
 *
 * ---
 *
 * ============================================================================
 * Plugin Parameters: Text Code Actions
 * ============================================================================
 *
 * Text codes are used for one of two things: performing actions or replacing
 * themselves with text data. This Plugin Parameter will focus on the aspect of
 * performing actions. These actions can be done through each JavaScript or by
 * a common event (if it is used in the Message Window). Adequate knowledge of
 * both is recommended before attempting to modify and/or add new Text Code
 * Actions to the Plugin Parameters.
 *
 * Each of the Text Code Actions are formatted in such a way:
 *
 * ---
 *
 * Text Code Action
 *
 *   Match:
 *   - This is what needs to be matched in order for this text code to work.
 *   - This is the primary text marker after the \ in a text code.
 *   - In \N[x], this would be the 'N'.
 *
 *   Type:
 *   - The type of parameter to obtain (none, number, or string).
 *   - This is the way the text code determines the condition type.
 *   - In \N[x], this would be the '[x]'.
 *
 *   Common Event:
 *   - Select a common event to run when this text code is used in a message.
 *
 *   JS: Action:
 *   - JavaScript code used to perform an action when this text code appears.
 *
 * ---
 *
 * ============================================================================
 * Plugin Parameters: Text Code Replacements
 * ============================================================================
 *
 * Text codes are used for one of two things: performing actions or replacing
 * themselves with text data. This Plugin Parameter will focus on the aspect of
 * replacing the text codes with text data. Text data can be replaced with
 * an exact exchange of text or dynamically through JavaScript. Adding a new
 * Text Code Replacement is done through the Plugin Parameters.
 *
 * Each of the Text Code Replacements are formatted in such a way:
 *
 * ---
 *
 * Text Code Replacement
 *
 *   Match:
 *   - This is what needs to be matched in order for this text code to work.
 *   - This is the primary text marker after the \ in a text code.
 *   - In \N[x], this would be the 'N'.
 *
 *   Type:
 *   - The type of parameter to obtain (none, number, or string).
 *   - This is the way the text code determines the condition type.
 *   - In \N[x], this would be the '[x]'.
 *
 *   STR: Text:
 *   - The text that will appear if this match appears.
 *     If this has a value, ignore the JS: Text version.
 *
 *   JS: Text:
 *   - JavaScript code used to determine the text that will appear if this
 *     match appears.
 *
 * ---
 *
 * ============================================================================
 * Plugin Parameters: Text Macros
 * ============================================================================
 *
 * Text macros are used in similar fashion to text codes replacements to
 * replace themselves with text data. The primary difference is that macros are
 * made in a different format with no conditional argument modifiers (ie the
 * [x] that follows a text code).
 *
 * To use a text macro, type in the matching keyword between two [brackets] and
 * it will be replaced by the string data or run the JavaScript code found in
 * the Plugin Parameter settings.
 *
 * For example, if you have the text macro "Leader", made to return the party
 * leader's name, you can type in [Leader] in the Message Window and it will be
 * replaced with the party leader's name. The output can also output text codes
 * into the resulting text.
 * 
 * This does NOT work with \MacroName as it did with Yanfly Engine Plugins.
 * Use the method stated before with the brackets to [MacroName] instead.
 *
 * Each of the Text Macros are formatted in such a way:
 *
 * ---
 *
 * Text Macro
 *
 *   Match:
 *   - This is what needs to be matched in order for this macro to work.
 *   - In [Leader], this would be the 'Leader' text.
 *
 *   STR: Text:
 *   - The replacement text that will appear from the macro.
 *   - If this has a value, ignore the JS: Text version.
 *
 *   JS: Text:
 *   - JavaScript code used to determine the text that will appear if this
 *     macro appears.
 *
 * ---
 *
 * ============================================================================
 * Plugin Parameters: Text Speed Option Settings
 * ============================================================================
 *
 * Modern RPG's on the market have the option to adjust the message speed rate
 * for players. These Plugin Parameters allow you to add that option to the
 * Options Menu as well.
 *
 * ---
 *
 * Text Speed Option Settings
 *
 *   Add Option?:
 *   - Add the 'Text Speed' option to the Options menu?
 *
 *   Adjust Window Height:
 *   - Automatically adjust the options window height?
 *
 *   Option Name:
 *   - Command name of the option.
 *
 *   Default Value:
 *   - 1 - 10, slowest to fastest.
 *   - 11 is instant value.
 *
 *   Instant Speed:
 *   - Text to show "instant" text.
 *
 * ---
 * 
 * ============================================================================
 * Plugin Parameters: Word Wrap Settings
 * ============================================================================
 *
 * Word wrap is a property that will cause any overflowing text to wrap around
 * and move into the next line. This property can only be enabled inside text
 * that accept text codes, such as the Message Window and Help Window. However,
 * word wrap is disabled for the Choice Window due to the nature of the Choice
 * Window's base properties.
 *
 * Word wrap can be enabled or disabled in three ways. One is by using the text
 * code <WordWrap> to enable it or </WordWrap> to disable it. The second method
 * is by enabling it with the Plugin Command: 'Message: Properties'. The third
 * method is by enabling it by default with the Plugin Parameters.
 *
 * ---
 *
 * Enable Word Wrap
 *
 *   Message Window:
 *   - Automatically enable Word Wrap for this window?
 *
 *   Help Window:
 *   - Automatically enable Word Wrap for this window?
 *
 * ---
 *
 * Rules
 *
 *   Link Break -> Space:
 *   - Convert manually placed (non tagged) line breaks with spaces?
 *   - Line breaks must be inserted using the <br> text code.
 *
 *   Tight Wrap:
 *   - If a face graphic is present in a message, word wrap will be tighter.
 * 
 *   End Padding:
 *   - Add extra padding to your window to make text wrap further away from the
 *     end of the window.
 *   - This will default to 0.
 *
 * ---
 *
 * ============================================================================
 * Terms of Use
 * ============================================================================
 *
 * 1. These plugins may be used in free or commercial games provided that they
 * have been acquired through legitimate means at VisuStella.com and/or any
 * other official approved VisuStella sources. Exceptions and special
 * circumstances that may prohibit usage will be listed on VisuStella.com.
 * 
 * 2. All of the listed coders found in the Credits section of this plugin must
 * be given credit in your games or credited as a collective under the name:
 * "VisuStella".
 * 
 * 3. You may edit the source code to suit your needs, so long as you do not
 * claim the source code belongs to you. VisuStella also does not take
 * responsibility for the plugin if any changes have been made to the plugin's
 * code, nor does VisuStella take responsibility for user-provided custom code
 * used for custom control effects including advanced JavaScript notetags
 * and/or plugin parameters that allow custom JavaScript code.
 * 
 * 4. You may NOT redistribute these plugins nor take code from this plugin to
 * use as your own. These plugins and their code are only to be downloaded from
 * VisuStella.com and other official/approved VisuStella sources. A list of
 * official/approved sources can also be found on VisuStella.com.
 *
 * 5. VisuStella is not responsible for problems found in your game due to
 * unintended usage, incompatibility problems with plugins outside of the
 * VisuStella MZ library, plugin versions that aren't up to date, nor
 * responsible for the proper working of compatibility patches made by any
 * third parties. VisuStella is not responsible for errors caused by any
 * user-provided custom code used for custom control effects including advanced
 * JavaScript notetags and/or plugin parameters that allow JavaScript code.
 *
 * 6. If a compatibility patch needs to be made through a third party that is
 * unaffiliated with VisuStella that involves using code from the VisuStella MZ
 * library, contact must be made with a member from VisuStella and have it
 * approved. The patch would be placed on VisuStella.com as a free download
 * to the public. Such patches cannot be sold for monetary gain, including
 * commissions, crowdfunding, and/or donations.
 *
 * ============================================================================
 * Credits
 * ============================================================================
 * 
 * If you are using this plugin, credit the following people in your game:
 * 
 * Team VisuStella
 * * Yanfly
 * * Arisu
 * * Olivia
 * * Irina
 *
 * ============================================================================
 * Changelog
 * ============================================================================
 * 
 * Version 1.27: October 7, 2021
 * * Optimization Update!
 * ** Plugin should run more optimized.
 * 
 * Version 1.26: September 3, 2021
 * * Bug Fixes!
 * ** Macros should now work properly with any \x<n> based text codes.
 *    Fix made by Irina.
 * 
 * Version 1.25: August 27, 2021
 * * Feature Update!
 * ** Macros should now work with the <WordWrap> text code. Update by Irina.
 * 
 * Version 1.24: August 20, 2021
 * * Feature Update!
 * ** Macros should now work with window placement and resize options.
 *    Update made by Irina.
 * ** Macros should now work with choice-related enable and visibility options.
 *    Update made by Irina.
 * 
 * Version 1.23: July 16, 2021
 * * Documentation Update!
 * ** Help file updated for new features.
 * * New Features!
 * ** New Plugin Parameter added by Irina:
 * *** Plugin Parameters > Word Wrap Settings > End Padding
 * **** Add extra padding to your window to make text wrap further away from
 *      the end of the window. This will default to 0.
 * 
 * Version 1.22: July 2, 2021
 * * Documentation Update!
 * ** Help file updated for new features.
 * * New Features!
 * ** New Text Codes added by Irina and sponsored by AndyL:
 * *** <Current Battle Target>
 * *** <Current Battle User>
 * **** Replaces the text code with the current target or current user's name
 *      in-battle. Otherwise, returns nothing.
 * **** Not recommended to be used inside of Help Descriptions. They are best
 *      used with "Show Text" event commands.
 * *** <Current Battle Action>
 * *** <Current Battle Action Name>
 * **** Replaces the text code with the current battle action's name with the
 *      icon or without it respectively. Otherwise, returns nothing.
 * **** Not recommended to be used inside of Help Descriptions. They are best
 *      used with "Show Text" event commands.
 * 
 * Version 1.21: June 4, 2021
 * * Documentation Update!
 * ** Added extra note to the new <Position: x, y, width, height> text codes
 *    that they do not work with Word Wrap.
 * * Feature Update!
 * ** Added fail safe for preventing Common Events that don't exist from being
 *    ran at all by the Message Window. Added by Arisu.
 * 
 * Version 1.20: May 28, 2021
 * * Documentation Update!
 * ** Help file updated for new features.
 * ** Added additional clarity for \WindowMoveTo<?> and \WindowMoveBy<?> and
 *    \WindowReset text codes with "Note 2".
 * *** Replace '?' with the following format: targetX, targetY, targetWidth,
 *     targetHeight, duration, easingType. Only targetX and targetY are
 *     required arguments. These will only alter the window dimensions when the
 *     text has arrived at that point. They will not alter the window
 *     preemptively. This is not used as a window positioner. Use the
 *     <Position: x, y, width, height> text code for that.
 * * New Features!
 * ** New hard-coded text codes added for Message Window Only. Added by Irina.
 * *** <Position: x, y, width, height>
 * *** <Coordinates: x, y>
 * *** <Dimensions: width, height>
 * 
 * Version 1.19: May 14, 2021
 * * Feature Updates!
 * ** <br> line breaks can now be used by Show Choices. Make sure that there is
 *    enough room to contain the text through Plugin Commands. Update by Irina.
 * 
 * Version 1.18: April 30, 2021
 * * Bug Fixes!
 * ** Moving windows with 0 duration via text code should now instantly move
 *    the windows to the desired location with no delay. Fix made by Olivia.
 * 
 * Version 1.17: April 9, 2021
 * * Feature Update!
 * ** <Auto> text codes for message windows will round up calculations for the
 *    message width to the nearest even number for better calculations.
 * 
 * Version 1.16: April 2, 2021
 * * Bug Fixes!
 * ** \CommonEvent[x] text code will no longer run upon message window size
 *    calculation. Fix made by Arisu.
 * * Documentation Update!
 * ** Added further clarification for "Text Macros" section.
 * *** This does NOT work with \MacroName as it did with Yanfly Engine Plugins.
 *     Use the method stated before with the brackets to [MacroName] instead.
 * 
 * Version 1.15: March 5, 2021
 * * Bug Fixes!
 * ** Hidden choices by switches will no longer count towards the maximum line
 *    count for Show Choice options. Fix made by Irina.
 * 
 * Version 1.14: February 12, 2021
 * * Bug Fixes!
 * ** Auto positioned messages in battle will no longer cover the battler in
 *    question. Fix made by Irina.
 * 
 * Version 1.13: February 5, 2021
 * * Bug Fixes!
 * ** Choice List Window with a dimmed background should now have a more
 *    consistent sized dim sprite. Fix made by Irina.
 * 
 * Version 1.12: January 22, 2021
 * * Feature Update!
 * ** Name Box Window Default Color is now disabled by default to 0 because
 *    users do not understand why their names are showing up yellow and did not
 *    bother reading the documentation. If users want this feature turned on,
 *    they will have to do it manually from now on. Update made by Irina.
 * 
 * Version 1.11: January 15, 2021
 * * Optimization Update!
 * ** Plugin should run more optimized.
 * 
 * Version 1.10: January 8, 2021
 * * Bug Fixes!
 * ** <Auto Actor: x> and <Auto Party: x> text codes should now work properly.
 *    Fix made by Irina.
 * * Feature Update!
 * ** Auto Color Plugin Parameters now have their default settings set to 0.
 *    This is due to an influx of "bug reports" from users who do not
 *    understand how this feature works, and the VisuStella team has decided it
 *    is better for the feature to default to an inactive state until users
 *    decide to search and utilize it themselves. Update made by Irina.
 * 
 * Version 1.09: January 1, 2021
 * * Feature Update!
 * ** Auto-color no longer applies to database names that are only numbers.
 *    Auto-color entries that are only numbers will also be ignored. This is to
 *    prevent breaking the text code parsing. Update made by Yanfly.
 * 
 * Version 1.08: November 15, 2020
 * * Documentation Update!
 * ** Some text codes left for the Name Box Window have been accidentally left
 *    out. These text codes allow for the positioning of the Name Box Window.
 *    Also, added to this section are the \NormalBG, \DimBG, and \TransparentBG
 *    text codes since people have been asking for how to change the name box
 *    window's background, but have skimmed over those text codes in different
 *    sections of the help file.
 * * Optimization Update!
 * ** Plugin should run more optimized.
 * 
 * Version 1.07: November 8, 2020
 * * Bug Fixes!
 * ** When using auto size functions, the message pause symbol will no longer
 *    appear semi-transparent the whole time. Fix made by Irina.
 * 
 * Version 1.06: October 25, 2020
 * * Documentation Update!
 * ** Added a warning message to the Fast Forward Key plugin parameter:
 * *** WARNING: If this key is the same as the dash button, this will clear out
 *     any held down inputs upon triggering an event  to prevent players from
 *     skipping potentially useful information stored in messages. If you do
 *     not want the input to be cleared, use a different key.
 * ** Updated help file for new features.
 * * Feature Update!
 * ** The default Fast Forward Key setting has now been changed from "Shift" to
 *    "Page Down". Change made by Yanfly
 * * New Feature!
 * ** New Plugin Parameter added by Irina.
 * *** Plugin Parameters > General > Default Outline Width
 * **** Changes the default outline width to this many pixels thick.
 * 
 * Version 1.06: September 27, 2020
 * * Bug Fixes!
 * ** Setting an actor's autocolor will now disable it from \N[x] and \P[x]
 *    text codes. Fix made by Irina.
 * 
 * Version 1.05: September 20, 2020
 * * Bug Fixes!
 * ** Auto Position text codes not place positions properly if the screen width
 *    and height differ from the box width and box height. Fix made by Irina.
 * 
 * Version 1.04: September 13, 2020
 * * Bug Fixes!
 * ** Word wrap no longer affects specific battle messages. Fix made by Irina.
 * ** Word wrap now updates properly after using the 'Message: Properties'
 *    Plugin Command. Fix made by Arisu.
 * 
 * Version 1.03: September 6, 2020
 * * Bug Fixes!
 * ** Autoplacement of the name box window now takes its offset Y setting into
 *    account before sending it to the bottom of the message window. Fix made
 *    by Yanfly.
 * ** Added automatic feature setting to turn off word wrap when using the
 *    auto-size and auto-position text codes. This is because the auto-size and
 *    auto-position effects don't work properly with Word Wrap based on how
 *    they both clash when adjusting the window settings. Fix made by Irina.
 * ** New message pages after auto-sizing no longer put out empty messages.
 *    Fix made by Irina and Shiro.
 * * Documentation Update!
 * ** Extended the note for auto-size and auto-position text codes to include
 *    that they do not work with Word Wrap. Added by Irina.
 * 
 * Version 1.02: August 30, 2020
 * * New Features!
 * ** Added new hard-coded text codes for auto-sizing and auto-positioning:
 * *** <Auto>, <Auto Width>, <Auto Height>
 * *** <Auto Actor: x>, <Auto Party: x>, <Auto Enemy: x>
 * *** <Auto Player>, <Auto Actor: x>, <Auto Party: x>, <Auto Event: x>
 * **** New features added by Irina.
 * 
 * Version 1.01: August 23, 2020
 * * Bug Fixes!
 * ** </Wordwrap> now works.
 * ** \ActorFace[x] text code now fixed.
 * *** Users updating from version 1.00 will need to fix this problem by either
 *     removing the plugin from the Plugin Manager list and reinstalling it, or
 *     going to Plugin Parameters > Text Code Replacements > ActorFace >
 *     JS: Text > and changing "$gameActors.actor(1)" to
 *     "$gameActors.actor(actorId)"
 * ** Actors with empty names would cause auto hightlight problems. Fixed!
 * ** Auto-colors now ignore names with special characters like !, ?, [, ], and
 *    so on.
 * ** Line break spacing fixed.
 * * New Features!
 * ** Wordwrap now works with <left>, <center> and <right> alignment tags.
 *
 * Version 1.00: August 20, 2020
 * * Finished Plugin!
 *
 * ============================================================================
 * End of Helpfile
 * ============================================================================
 *
 * @ --------------------------------------------------------------------------
 *
 * @command MessageWindowProperties
 * @text Message: Properties
 * @desc Change the various properties of the Message Window.
 *
 * @arg Rows:num
 * @text Rows
 * @type number
 * @min 0
 * @desc Change the number of Message Window rows.
 * Leave at 0 to keep it unchanged.
 * @default 4
 *
 * @arg Width:num
 * @text Width
 * @type number
 * @min 0
 * @desc Change the Message Window width in pixels.
 * Leave at 0 to keep it unchanged.
 * @default 816
 *
 * @arg Center:eval
 * @text Center Window X?
 * @parent Width
 * @type boolean
 * @on Center
 * @off Don't
 * @desc Center the window X after changing its width?
 * @default true
 *
 * @arg WordWrap:str
 * @text Word Wrap
 * @type select
 * @option No Change
 * @value No Change
 * @option Enable
 * @value true
 * @option Disable
 * @value false
 * @desc Enable or disable Word Wrap for the Message Window?
 * @default No Change
 *
 * @ --------------------------------------------------------------------------
 *
 * @command ChoiceWindowProperties
 * @text Choices: Properties
 * @desc Change the properties found in the Show Choices event command.
 *
 * @arg LineHeight:num
 * @text Line Height
 * @type number
 * @min 0
 * @desc Change the line height for the show choices.
 * Leave at 0 to keep this unchanged.
 * @default 36
 *
 * @arg MaxRows:num
 * @text Max Rows
 * @type number
 * @min 0
 * @desc Maximum number of choice rows to be displayed.
 * Leave at 0 to keep this unchanged.
 * @default 8
 *
 * @arg MaxCols:num
 * @text Max Columns
 * @type number
 * @min 0
 * @desc Maximum number of choice columns to be displayed.
 * Leave at 0 to keep this unchanged.
 * @default 1
 *
 * @arg TextAlign:str
 * @text Text Alignment
 * @type select
 * @option Default
 * @value default
 * @option Left
 * @value left
 * @option Center
 * @value center
 * @option Right
 * @value right
 * @desc Text alignment for Show Choice window.
 * @default default
 *
 * @ --------------------------------------------------------------------------
 *
 * @ ==========================================================================
 * @ Plugin Parameters
 * @ ==========================================================================
 *
 * @param BreakHead
 * @text --------------------------
 * @default ----------------------------------
 *
 * @param MessageCore
 * @default Plugin Parameters
 *
 * @param ATTENTION
 * @default READ THE HELP FILE
 *
 * @param BreakSettings
 * @text --------------------------
 * @default ----------------------------------
 *
 * @param General:struct
 * @text General Settings
 * @type struct<General>
 * @desc General settings involving the message system.
 * @default {"MessageWindow":"","MessageRows:num":"4","MessageWidth:num":"816","FastForwardKey:str":"pagedown","MessageTextDelay:num":"1","StretchDimmedBg:eval":"true","DefaultOutlineWidth:num":"3","NameBoxWindow":"","NameBoxWindowDefaultColor:num":"0","NameBoxWindowOffsetX:num":"0","NameBoxWindowOffsetY:num":"0","ChoiceListWindow":"","ChoiceWindowLineHeight:num":"36","ChoiceWindowMaxRows:num":"8","ChoiceWindowMaxCols:num":"1","ChoiceWindowTextAlign:str":"default","DefaultTextCodes":"","RelativePXPY:eval":"true","FontBiggerCap:eval":"108","FontSmallerCap:eval":"12","FontChangeValue:eval":"12"}
 *
 * @param AutoColor:struct
 * @text Auto-Color Settings
 * @type struct<AutoColor>
 * @desc Automatically color certain keywords a specific way.
 * @default {"DatabaseHighlighting":"","Actors:str":"0","Classes:str":"0","Skills:str":"0","Items:str":"0","Weapons:str":"0","Armors:str":"0","Enemies:str":"0","States:str":"0","WordHighlighting":"","TextColor1:arraystr":"[]","TextColor2:arraystr":"[]","TextColor3:arraystr":"[]","TextColor4:arraystr":"[]","TextColor5:arraystr":"[]","TextColor6:arraystr":"[]","TextColor7:arraystr":"[]","TextColor8:arraystr":"[]","TextColor9:arraystr":"[]","TextColor10:arraystr":"[]","TextColor11:arraystr":"[]","TextColor12:arraystr":"[]","TextColor13:arraystr":"[]","TextColor14:arraystr":"[]","TextColor15:arraystr":"[]","TextColor16:arraystr":"[]","TextColor17:arraystr":"[]","TextColor18:arraystr":"[]","TextColor19:arraystr":"[]","TextColor20:arraystr":"[]","TextColor21:arraystr":"[]","TextColor22:arraystr":"[]","TextColor23:arraystr":"[]","TextColor24:arraystr":"[]","TextColor25:arraystr":"[]","TextColor26:arraystr":"[]","TextColor27:arraystr":"[]","TextColor28:arraystr":"[]","TextColor29:arraystr":"[]","TextColor30:arraystr":"[]","TextColor31:arraystr":"[]"}
 *
 * @param TextCodeActions:arraystruct
 * @text Text Code Actions
 * @type struct<TextCodeAction>[]
 * @desc Text codes that perform actions.
 * @default ["{\"Match:str\":\"ChangeFace\",\"Type:str\":\"\\\\<(.*?)\\\\>\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst data = this.obtainEscapeString(textState).split(',');\\\\nif (textState.drawing) {\\\\n    const filename = data[0].trim();\\\\n    const index = parseInt(data[1] || '0');\\\\n    $gameMessage.setFaceImage(filename, index);\\\\n    this.loadMessageFace();\\\\n    const rtl = $gameMessage.isRTL();\\\\n    const width = ImageManager.faceWidth;\\\\n    const height = this.innerHeight;\\\\n    const x = rtl ? this.innerWidth - width - 4 : 4;\\\\n    this.contents.clearRect(x, 0, width, height);\\\\n    this._faceBitmap.addLoadListener(this.drawMessageFace.bind(this));\\\\n}\\\"\"}","{\"Match:str\":\"FaceIndex\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst index = this.obtainEscapeParam(textState);\\\\nif (textState.drawing) {\\\\n    const filename = $gameMessage.faceName();\\\\n    $gameMessage.setFaceImage(filename, index);\\\\n    this.loadMessageFace();\\\\n    const rtl = $gameMessage.isRTL();\\\\n    const width = ImageManager.faceWidth;\\\\n    const height = this.innerHeight;\\\\n    const x = rtl ? this.innerWidth - width - 4 : 4;\\\\n    this.contents.clearRect(x, 0, width, height);\\\\n    this._faceBitmap.addLoadListener(this.drawMessageFace.bind(this));\\\\n}\\\"\"}","{\"Match:str\":\"TextDelay\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst delay = this.obtainEscapeParam(textState);\\\\nif (textState.drawing && this.constructor === Window_Message) {\\\\n    this.setTextDelay(delay);\\\\n}\\\"\"}","{\"Match:str\":\"NormalBG\",\"Type:str\":\"\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nif (textState.drawing) {\\\\n    this.setBackgroundType(0);\\\\n}\\\"\"}","{\"Match:str\":\"DimBG\",\"Type:str\":\"\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nif (textState.drawing) {\\\\n    this.setBackgroundType(1);\\\\n}\\\"\"}","{\"Match:str\":\"TransparentBG\",\"Type:str\":\"\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nif (textState.drawing) {\\\\n    this.setBackgroundType(2);\\\\n}\\\"\"}","{\"Match:str\":\"FontChange\",\"Type:str\":\"\\\\<(.*?)\\\\>\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst fontName = this.obtainEscapeString(textState);\\\\nthis.contents.fontFace = fontName;\\\"\"}","{\"Match:str\":\"ResetFont\",\"Type:str\":\"\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"this.resetFontSettings();\\\"\"}","{\"Match:str\":\"ResetColor\",\"Type:str\":\"\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"this.resetTextColor();\\\"\"}","{\"Match:str\":\"HexColor\",\"Type:str\":\"\\\\<(.*?)\\\\>\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst hexColor = this.obtainEscapeString(textState);\\\\nif (!this.isColorLocked() && textState.drawing) {\\\\n    this.changeTextColor(hexColor);\\\\n}\\\"\"}","{\"Match:str\":\"OutlineColor\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst colorIndex = this.obtainEscapeParam(textState);\\\\nif (!this.isColorLocked() && textState.drawing) {\\\\n    this.changeOutlineColor(ColorManager.textColor(colorIndex));\\\\n}\\\"\"}","{\"Match:str\":\"OutlineHexColor\",\"Type:str\":\"\\\\<(.*?)\\\\>\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst hexColor = this.obtainEscapeString(textState);\\\\nif (!this.isColorLocked() && textState.drawing) {\\\\n    this.changeOutlineColor(hexColor);\\\\n}\\\"\"}","{\"Match:str\":\"OutlineWidth\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst width = this.obtainEscapeParam(textState);\\\\nif (textState.drawing) {\\\\n    this.contents.outlineWidth = width;\\\\n}\\\"\"}","{\"Match:str\":\"WindowMoveTo\",\"Type:str\":\"\\\\<(.*?)\\\\>\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst data = this.obtainEscapeString(textState).split(',');\\\\nif (textState.drawing) {\\\\n    const x = !!data[0] ? Number(data[0].trim()) : this.x;\\\\n    const y = !!data[1] ? Number(data[1].trim()) : this.y;\\\\n    const width = !!data[2] ? Number(data[2].trim()) : this.width;\\\\n    const height = !!data[3] ? Number(data[3].trim()) : this.height;\\\\n    const duration = !!data[4] ? Number(data[4].trim()) : 20;\\\\n    const easingType = !!data[5] ? data[5].trim() : 0;\\\\n    this.moveTo(x, y, width, height, duration, easingType);\\\\n}\\\"\"}","{\"Match:str\":\"WindowMoveBy\",\"Type:str\":\"\\\\<(.*?)\\\\>\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nconst data = this.obtainEscapeString(textState).split(',');\\\\nif (textState.drawing) {\\\\n    const x = !!data[0] ? Number(data[0].trim()) : 0;\\\\n    const y = !!data[1] ? Number(data[1].trim()) : 0;\\\\n    const width = !!data[2] ? Number(data[2].trim()) : 0;\\\\n    const height = !!data[3] ? Number(data[3].trim()) : 0;\\\\n    const duration = !!data[4] ? Number(data[4].trim()) : 20;\\\\n    const easingType = !!data[5] ? data[5].trim() : 0;\\\\n    this.moveBy(x, y, width, height, duration, easingType);\\\\n}\\\"\"}","{\"Match:str\":\"WindowReset\",\"Type:str\":\"\",\"CommonEvent:num\":\"0\",\"ActionJS:func\":\"\\\"const textState = arguments[0];\\\\nif (textState.drawing) {\\\\n    const frames = 20;\\\\n    const easingType = 0;\\\\n    this.resetRect(frames, easingType);\\\\n}\\\"\"}"]
 *
 * @param TextCodeReplace:arraystruct
 * @text Text Code Replacements
 * @type struct<TextCodeReplace>[]
 * @desc Text codes that replace themselves with text.
 * @default ["{\"Match:str\":\"ActorFace\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const actorId = parseInt(arguments[1]);\\\\nconst actor = $gameActors.actor(actorId);\\\\nif (this.constructor === Window_Message && actor) {\\\\n    $gameMessage.setFaceImage(\\\\n        actor.faceName(),\\\\n        actor.faceIndex()\\\\n    );\\\\n}\\\\nreturn '';\\\"\"}","{\"Match:str\":\"PartyFace\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const index = parseInt(arguments[1]) - 1;\\\\nconst actor = $gameParty.members()[index];\\\\nif (this.constructor === Window_Message && actor) {\\\\n    $gameMessage.setFaceImage(\\\\n        actor.faceName(),\\\\n        actor.faceIndex()\\\\n    );\\\\n}\\\\nreturn '';\\\"\"}","{\"Match:str\":\"Class\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataClasses;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"ClassName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataClasses;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"Skill\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataSkills;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"SkillName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataSkills;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"Item\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataItems;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"ItemName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataItems;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"ItemQuantity\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataItems;\\\\nconst id = parseInt(arguments[1]);\\\\nreturn $gameParty.numItems(database[id]);\\\"\"}","{\"Match:str\":\"Weapon\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataWeapons;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"WeaponName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataWeapons;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"WeaponQuantity\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataWeapons;\\\\nconst id = parseInt(arguments[1]);\\\\nreturn $gameParty.numItems(database[id]);\\\"\"}","{\"Match:str\":\"LastGainObj\",\"Type:str\":\"\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const icon = true;\\\\nreturn this.lastGainedObjectName(icon);\\\"\"}","{\"Match:str\":\"LastGainObjName\",\"Type:str\":\"\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const icon = false;\\\\nreturn this.lastGainedObjectName(icon);\\\"\"}","{\"Match:str\":\"LastGainObjQuantity\",\"Type:str\":\"\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"return this.lastGainedObjectQuantity();\\\"\"}","{\"Match:str\":\"Armor\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataArmors;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"ArmorName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataArmors;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"ArmorQuantity\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataArmors;\\\\nconst id = parseInt(arguments[1]);\\\\nreturn $gameParty.numItems(database[id]);\\\"\"}","{\"Match:str\":\"State\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataStates;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"StateName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataStates;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"Enemy\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataEnemies;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"EnemyName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataEnemies;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"Troop\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataTroops;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"TroopName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"const database = $dataTroops;\\\\nconst id = parseInt(arguments[1]);\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"TroopMember\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"if (!$gameParty.inBattle()) return \\\\\\\"\\\\\\\";\\\\nconst index = (parseInt(arguments[1]) - 1) || 0;\\\\nconst member = $gameTroop.members()[index];\\\\nconst database = $dataEnemies;\\\\nconst id = member ? member.enemyId() : 0;\\\\nconst icon = true;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}","{\"Match:str\":\"TroopMemberName\",\"Type:str\":\"\\\\[(\\\\d+)\\\\]\",\"TextStr:str\":\"Undefined\",\"TextJS:func\":\"\\\"if (!$gameParty.inBattle()) return \\\\\\\"\\\\\\\";\\\\nconst index = (parseInt(arguments[1]) - 1) || 0;\\\\nconst member = $gameTroop.members()[index];\\\\nconst database = $dataEnemies;\\\\nconst id = member ? member.enemyId() : 0;\\\\nconst icon = false;\\\\nreturn this.databaseObjectName(database, id, icon);\\\"\"}"]
 *
 * @param TextMacros:arraystruct
 * @text Text Macros
 * @type struct<TextMacro>[]
 * @desc Macros that are used to quickly write batches of text.
 * @default ["{\"Match:str\":\"Example Macro\",\"TextStr:str\":\"This is the text that will be displayed when you type [Example Macro].\",\"TextJS:func\":\"\\\"return 'Text';\\\"\"}","{\"Match:str\":\"Leader\",\"TextStr:str\":\"\\\\P[1]\",\"TextJS:func\":\"\\\"return 'Text';\\\"\"}"]
 *
 * @param TextSpeed:struct
 * @text Text Speed Option Settings
 * @type struct<TextSpeed>
 * @desc Text Speed Options Menu settings.
 * @default {"AddOption:eval":"true","AdjustRect:eval":"true","Name:str":"Text Speed","Default:num":"10","Instant:str":"Instant"}
 *
 * @param WordWrap:struct
 * @text Word Wrap Settings
 * @type struct<WordWrap>
 * @desc Settings involving Word Wrap.
 * @default {"EnableWordWrap":"","MessageWindow:eval":"false","HelpWindow:eval":"false","Rules":"","LineBreakSpace:eval":"true","TightWrap:eval":"false","EndPadding:num":"0"}
 *
 * @param BreakEnd1
 * @text --------------------------
 * @default ----------------------------------
 *
 * @param End Of
 * @default Plugin Parameters
 *
 * @param BreakEnd2
 * @text --------------------------
 * @default ----------------------------------
 *
 */
/* ----------------------------------------------------------------------------
 * General Settings
 * ----------------------------------------------------------------------------
 */
/*~struct~General:
 *
 * @param MessageWindow
 * @text Message Window
 *
 * @param MessageRows:num
 * @text Default Rows
 * @parent MessageWindow
 * @type num
 * @min 1
 * @desc Default number of rows to display for the Message Window.
 * @default 4
 *
 * @param MessageWidth:num
 * @text Default Width
 * @parent MessageWindow
 * @type num
 * @min 1
 * @desc Default Message Window width in pixels.
 * @default 816
 *
 * @param FastForwardKey:str
 * @text Fast Forward Key
 * @parent MessageWindow
 * @type combo
 * @option none
 * @option tab
 * @option shift
 * @option control
 * @option pageup
 * @option pagedown
 * @desc This is the key used for fast forwarding messages.
 * @default pagedown
 *
 * @param MessageTextDelay:num
 * @text Text Delay
 * @parent MessageWindow
 * @type number
 * @min 0
 * @desc How many frames to wait between characters drawn?
 * Use 0 for instant.
 * @default 1
 *
 * @param StretchDimmedBg:eval
 * @text Stretch Dimmed BG
 * @parent MessageWindow
 * @type boolean
 * @on Stretch
 * @off Don't
 * @desc Stretch dimmed window background to fit the whole screen.
 * @default true
 *
 * @param DefaultOutlineWidth:num
 * @text Default Outline Width
 * @parent MessageWindow
 * @type number
 * @min 0
 * @desc Changes the default outline width to this many pixels thick.
 * @default 3
 *
 * @param NameBoxWindow
 * @text Name Box Window
 *
 * @param NameBoxWindowDefaultColor:num
 * @text Default Color
 * @parent NameBoxWindow
 * @min 0
 * @max 31
 * @desc Default color for the Name Box Window's text.
 * @default 0
 *
 * @param NameBoxWindowOffsetX:num
 * @text Offset X
 * @parent NameBoxWindow
 * @desc How much to offset the name box window X by (as long as it doesn't go offscreen).
 * @default 0
 *
 * @param NameBoxWindowOffsetY:num
 * @text Offset Y
 * @parent NameBoxWindow
 * @desc How much to offset the name box window Y by (as long as it doesn't go offscreen).
 * @default 0
 *
 * @param ChoiceListWindow
 * @text Choice List Window
 *
 * @param ChoiceWindowLineHeight:num
 * @text Line Height
 * @parent ChoiceListWindow
 * @type number
 * @min 1
 * @desc What is the default line height for Show Choices?
 * @default 36
 *
 * @param ChoiceWindowMaxRows:num
 * @text Max Rows
 * @parent ChoiceListWindow
 * @type number
 * @min 1
 * @desc Maximum number of rows to visibly display?
 * @default 8
 *
 * @param ChoiceWindowMaxCols:num
 * @text Max Columns
 * @parent ChoiceListWindow
 * @type number
 * @min 1
 * @desc Maximum number of columns to visibly display?
 * @default 1
 *
 * @param ChoiceWindowTextAlign:str
 * @text Text Alignment
 * @parent ChoiceListWindow
 * @type select
 * @option Default
 * @value default
 * @option Left
 * @value left
 * @option Center
 * @value center
 * @option Right
 * @value right
 * @desc Default alignment for Show Choice window.
 * @default default
 *
 * @param DefaultTextCodes
 * @text Default Text Codes
 *
 * @param RelativePXPY:eval
 * @text Relative \PX \PY
 * @parent DefaultTextCodes
 * @type boolean
 * @on Better
 * @off Normal
 * @desc Make \PX[x] and \PY[x] adjust relative starting position than exact coordinates.
 * @default true
 *
 * @param FontBiggerCap:eval
 * @text \{ Maximum
 * @parent DefaultTextCodes
 * @type number
 * @min 1
 * @desc Determine the maximum size that \{ can reach.
 * @default 108
 *
 * @param FontSmallerCap:eval
 * @text \} Minimum
 * @parent DefaultTextCodes
 * @type number
 * @min 1
 * @desc Determine the minimum size that \} can reach.
 * @default 12
 *
 * @param FontChangeValue:eval
 * @text \{ Change \}
 * @parent DefaultTextCodes
 * @type number
 * @min 1
 * @desc How much does \{ and \} change font size by?
 * @default 12
 *
 */
/* ----------------------------------------------------------------------------
 * Auto Color Settings
 * ----------------------------------------------------------------------------
 */
/*~struct~AutoColor:
 *
 * @param DatabaseHighlighting
 * @text Database Highlighting
 *
 * @param Actors:str
 * @text Actors
 * @parent DatabaseHighlighting
 * @type number
 * @min 0
 * @max 31
 * @desc Any usage of an Actor's name is given this text color.
 * Use 0 to not auto-color.
 * @default 0
 *
 * @param Classes:str
 * @text Classes
 * @parent DatabaseHighlighting
 * @type number
 * @min 0
 * @max 31
 * @desc Any usage of a Class's name is given this text color.
 * Use 0 to not auto-color.
 * @default 0
 *
 * @param Skills:str
 * @text Skills
 * @parent DatabaseHighlighting
 * @type number
 * @min 0
 * @max 31
 * @desc Any usage of a Skill's name is given this text color.
 * Use 0 to not auto-color.
 * @default 0
 *
 * @param Items:str
 * @text Items
 * @parent DatabaseHighlighting
 * @type number
 * @min 0
 * @max 31
 * @desc Any usage of an Item's name is given this text color.
 * Use 0 to not auto-color.
 * @default 0
 *
 * @param Weapons:str
 * @text Weapons
 * @parent DatabaseHighlighting
 * @type number
 * @min 0
 * @max 31
 * @desc Any usage of a Weapon's name is given this text color.
 * Use 0 to not auto-color.
 * @default 0
 *
 * @param Armors:str
 * @text Armors
 * @parent DatabaseHighlighting
 * @type number
 * @min 0
 * @max 31
 * @desc Any usage of an Armor's name is given this text color.
 * Use 0 to not auto-color.
 * @default 0
 *
 * @param Enemies:str
 * @text Enemies
 * @parent DatabaseHighlighting
 * @type number
 * @min 0
 * @max 31
 * @desc Any usage of an Enemy's name is given this text color.
 * Use 0 to not auto-color.
 * @default 0
 *
 * @param States:str
 * @text States
 * @parent DatabaseHighlighting
 * @type number
 * @min 0
 * @max 31
 * @desc Any usage of a State's name is given this text color.
 * Use 0 to not auto-color.
 * @default 0
 *
 * @param WordHighlighting
 * @text Word Highlighting
 *
 * @param TextColor1:arraystr
 * @text \C[1]: Blue
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor2:arraystr
 * @text \C[2]: Red
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor3:arraystr
 * @text \C[3]: Green
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor4:arraystr
 * @text \C[4]: Sky Blue
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor5:arraystr
 * @text \C[5]: Purple
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor6:arraystr
 * @text \C[6]: Yellow
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor7:arraystr
 * @text \C[7]: Gray
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor8:arraystr
 * @text \C[8]: Light Gray
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor9:arraystr
 * @text \C[9]: Dark Blue
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor10:arraystr
 * @text \C[10]: Dark Red
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor11:arraystr
 * @text \C[11]: Dark Green
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor12:arraystr
 * @text \C[12]: Dark Sky Blue
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor13:arraystr
 * @text \C[13]: Dark Purple
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor14:arraystr
 * @text \C[14]: Solid Yellow
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor15:arraystr
 * @text \C[15]: Black
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor16:arraystr
 * @text \C[16]: System Blue
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor17:arraystr
 * @text \C[17]: Crisis Yellow
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor18:arraystr
 * @text \C[18]: Dead Red
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor19:arraystr
 * @text \C[19]: Outline Black
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor20:arraystr
 * @text \C[20]: HP Orange 1
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor21:arraystr
 * @text \C[21]: HP Orange 2
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor22:arraystr
 * @text \C[22]: MP Blue 1
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor23:arraystr
 * @text \C[23]: MP Blue 2
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor24:arraystr
 * @text \C[24]: Param Up Green
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor25:arraystr
 * @text \C[25]: Param Down Red
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor26:arraystr
 * @text \C[26]: System Purple
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor27:arraystr
 * @text \C[27]: System Pink
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor28:arraystr
 * @text \C[28]: TP Green 1
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor29:arraystr
 * @text \C[29]: TP Green 2
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor30:arraystr
 * @text \C[30]: EXP Purple 1
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 * @param TextColor31:arraystr
 * @text \C[31]: EXP Purple 2
 * @parent WordHighlighting
 * @type string[]
 * @desc A list of all the words that will be automatically colored with this text color.
 * @default []
 *
 */
/* ----------------------------------------------------------------------------
 * Text Code Actions
 * ----------------------------------------------------------------------------
 */
/*~struct~TextCodeAction:
 *
 * @param Match:str
 * @text Match
 * @desc This is what needs to be matched in order for this text code to work.
 * @default Key
 *
 * @param Type:str
 * @text Type
 * @type select
 * @option none
 * @value 
 * @option [x] (number)
 * @value \[(\d+)\]
 * @option <x> (string)
 * @value \<(.*?)\>
 * @desc The type of parameter to obtain (none, number, or string).
 * @default 
 *
 * @param CommonEvent:num
 * @text Common Event
 * @type common_event
 * @desc Select a common event to run when this text code is used in a message.
 * @default 0
 *
 * @param ActionJS:func
 * @text JS: Action
 * @type note
 * @desc JavaScript code used to perform an action when this text code appears.
 * @default "const textState = arguments[0];"
 *
 */
/* ----------------------------------------------------------------------------
 * Text Code Replacements
 * ----------------------------------------------------------------------------
 */
/*~struct~TextCodeReplace:
 *
 * @param Match:str
 * @text Match
 * @desc This is what needs to be matched in order for this text code to work.
 * @default Key
 *
 * @param Type:str
 * @text Type
 * @type select
 * @option none
 * @value 
 * @option [x] (number)
 * @value \[(\d+)\]
 * @option <x> (string)
 * @value \<(.*?)\>
 * @desc The type of parameter to obtain (none, number, or string).
 * @default 
 *
 * @param TextStr:str
 * @text STR: Text
 * @desc The text that will appear if this match appears.
 * If this has a value, ignore the JS: Text version.
 * @default Undefined
 *
 * @param TextJS:func
 * @text JS: Text
 * @type note
 * @desc JavaScript code used to determine the text that will appear if this match appears.
 * @default "return 'Text';"
 *
 */
/* ----------------------------------------------------------------------------
 * Text Macro
 * ----------------------------------------------------------------------------
 */
/*~struct~TextMacro:
 *
 * @param Match:str
 * @text Match
 * @desc This is what needs to be matched in order for this macro to work.
 * @default Key
 *
 * @param TextStr:str
 * @text STR: Text
 * @desc The replacement text that will appear from the macro.
 * If this has a value, ignore the JS: Text version.
 * @default Undefined
 *
 * @param TextJS:func
 * @text JS: Text
 * @type note
 * @desc JavaScript code used to determine the text that will appear if this macro appears.
 * @default "return 'Text';"
 *
 */
/* ----------------------------------------------------------------------------
 * Text Speed Options Settings
 * ----------------------------------------------------------------------------
 */
/*~struct~TextSpeed:
 *
 * @param AddOption:eval
 * @text Add Option?
 * @type boolean
 * @on Add
 * @off Don't Add
 * @desc Add the 'Text Speed' option to the Options menu?
 * @default true
 *
 * @param AdjustRect:eval
 * @text Adjust Window Height
 * @type boolean
 * @on Adjust
 * @off Don't
 * @desc Automatically adjust the options window height?
 * @default true
 *
 * @param Name:str
 * @text Option Name
 * @desc Command name of the option.
 * @default Text Speed
 *
 * @param Default:num
 * @text Default Value
 * @type number
 * @min 1
 * @max 11
 * @desc 1 - 10, slowest to fastest.
 * 11 is instant value.
 * @default 10
 *
 * @param Instant:str
 * @text Instant Speed
 * @desc Text to show "instant" text.
 * @default Instant
 *
 */
/* ----------------------------------------------------------------------------
 * Word Wrap Settings
 * ----------------------------------------------------------------------------
 */
/*~struct~WordWrap:
 *
 * @param EnableWordWrap
 * @text Enable Word Wrap
 *
 * @param MessageWindow:eval
 * @text Message Window
 * @parent EnableWordWrap
 * @type boolean
 * @on Enable
 * @off Disable
 * @desc Automatically enable Word Wrap for this window?
 * @default false
 *
 * @param HelpWindow:eval
 * @text Help Window
 * @parent EnableWordWrap
 * @type boolean
 * @on Enable
 * @off Disable
 * @desc Automatically enable Word Wrap for this window?
 * @default false
 *
 * @param Rules
 * @text Rules
 *
 * @param LineBreakSpace:eval
 * @text Link Break -> Space
 * @parent Rules
 * @type boolean
 * @on Enable
 * @off Disable
 * @desc Convert manually placed (non tagged) line breaks with spaces?
 * @default true
 *
 * @param TightWrap:eval
 * @text Tight Wrap
 * @parent Rules
 * @type boolean
 * @on Enable
 * @off Disable
 * @desc If a face graphic is present in a message, word wrap will be tighter.
 * @default false
 *
 * @param EndPadding:num
 * @text End Padding
 * @parent Rules
 * @type number
 * @desc Add extra padding to your window to make text wrap further away from the end of the window.
 * @default 0
 *
 */
//=============================================================================

const _0x22ff2b=_0x1467;(function(_0x1dacfb,_0x1f5533){const _0x54e6dc=_0x1467,_0x1036cd=_0x1dacfb();while(!![]){try{const _0x1aec92=parseInt(_0x54e6dc(0x351))/0x1*(-parseInt(_0x54e6dc(0x256))/0x2)+parseInt(_0x54e6dc(0x2dd))/0x3+parseInt(_0x54e6dc(0x418))/0x4+parseInt(_0x54e6dc(0x2ef))/0x5+parseInt(_0x54e6dc(0x3a0))/0x6*(-parseInt(_0x54e6dc(0x41d))/0x7)+-parseInt(_0x54e6dc(0x1cd))/0x8+-parseInt(_0x54e6dc(0x1ce))/0x9;if(_0x1aec92===_0x1f5533)break;else _0x1036cd['push'](_0x1036cd['shift']());}catch(_0xe802d4){_0x1036cd['push'](_0x1036cd['shift']());}}}(_0x1363,0x884a3));var label=_0x22ff2b(0x3d5),tier=tier||0x0,dependencies=[],pluginData=$plugins['filter'](function(_0x4d5595){const _0x1a48a4=_0x22ff2b;return _0x4d5595[_0x1a48a4(0x40b)]&&_0x4d5595[_0x1a48a4(0x39b)]['includes']('['+label+']');})[0x0];function _0x1363(){const _0x69b930=['jyTIP','FontBiggerCap','padding','DefaultOutlineWidth','update','boxWidth','blt','textWidth','isMessageWindowWordWrap','outputHeight','drawBackPicture','wJGkb','ParseEnemyNotetags','ConvertParams','round','CreateAutoColorRegExpListEntries','</LEFT>','MessageTextDelay','<RIGHT>','createTextState','Match','isBreakShowTextCommands','gainItem','setup','call','postFlushTextState','applyMoveEasing','_dimmerSprite','yYdoL','ParseArmorNotetags','YaJVP','Width','remove','PREVCOLOR','iZGTY','sxFrP','convertChoiceMacros','returnPreservedFontSettings','value',')))','processTextAlignmentChange','height','partyMemberName','TextMacros','jELZI','LAvjS','battleActionName','Window_Message_updatePlacement','addWrapBreakAfterPunctuation','choices','heZvv','WORD_WRAP_PADDING','parseChoiceText','_resetRect','_moveTargetX','LniZC','map\x20actor','calcWindowHeight','toUpperCase','_messageWindow','getChoiceListMaxColumns','\x1bTEXTALIGNMENT[2]','Window_Message_clearFlags','TextCodeReplace','processDrawPicture','obtainEscapeParam','textSpeedStatusText','\x1bBOLD[1]','mhxTL','processActorNameAutoColorChanges','HohNV','refresh','startY','GDMXg','isHelpWindowWordWrap','Window_Base_processNewLine','MaxRows','ConfigManager_applyData','JSON','QZJjg','OJgdv','processPreviousColor','setWaitMode','scale','iconIndex','Window_Options_statusText','getPreservedFontSettings','initialize','_messageCommonEvents','processTextAlignmentX','nextEventCode','actor','itemHeight','process_VisuMZ_MessageCore_AutoColor','AxVmO','substring','battle\x20enemy','min','WRAPBREAK','WCwwm','processNewLine','getChoiceListLineHeight','_messagePositionReset','uzDvT','_textColorStack','max','vTyci','moveTo','prepareForcedPositionEscapeCharacters','cBUca','setupNumInput','battle\x20party','General','floor','canMove','3315132FOUSjb','IDVLk','registerCommand','map\x20party','split','BCqJf','currencyUnit','postConvertEscapeCharacters','mDAYh','UbBbd','convertNewPageTextStateMacros','setupEvents','changeOutlineColor','textCodeCheck','ParseItemNotetags','TextCodeActions','newPage','TextColor','1096450CbMjST','_colorLock','drawTextEx','adjustShowChoiceExtension','choiceTextAlign','COLORLOCK','convertBackslashCharacters','ParseAllNotetags','easeOut','mainFontSize','updateAutoSizePosition','aFMsF','currentCommand','Window_Help_refresh','addedWidth','length','\x1bITALIC[1]','changeTextColor','clearActorNameAutoColor','resetPositionX','Window_Base_processAllText','preConvertEscapeCharacters','ParseWeaponNotetags','\x1bTEXTALIGNMENT','isContinuePrepareShowTextCommands','_autoColorActorNames','normalColor','NameBoxWindowOffsetX','\x1bBOLD[0]','process_VisuMZ_MessageCore_TextMacros','_forcedPosition','processCustomWait','convertTextAlignmentEscapeCharacters','updateTransform','makeData','registerActorNameAutoColorChanges','Window_Message_terminateMessage','</B>','convertFontSettingsEscapeCharacters','fontBold','myQTs','obtainEscapeString','ftCnk','Window_Base_processControlCharacter','oCWrm','wAsZF','defeat','surprise','setTextAlignment','PTJNH','map\x20event','</RIGHT>','textCodeResult','\x1bi[%1]%2','processWrapBreak','return\x20\x27','onDatabaseLoaded','message','LBkvT','_moveTargetWidth','adjustShowChoiceCancel','fDRrO','setColorLock','processColorLock','Abmxo','zncPO','processPyTextCode','messagePositionReset','getConfigValue','vUFDf','sort','_texts','_lastGainedItemData','TextJS','\x1bTEXTALIGNMENT[0]','obtainExp','resetWordWrap','ParseStateNotetags','ARRAYJSON','moveBy','addContinuousShowTextCommands','EVAL','exit','refreshDimmerBitmap','WAIT','updateMove','ARRAYEVAL','Scene_Options_maxCommands','_list','AddAutoColor','<I>','registerResetRect','VzTfs','getTextAlignment','addGeneralOptions','unshift','actorName','setLastGainedItemData','8329CIqMId','_eventId','Oymvh','processControlCharacter','isVolumeSymbol','MaxCols','maxChoiceWidth','Game_Party_gainItem','_target','isSceneBattle','<LEFT>','_textDelay','initMessageCore','lastGainedObjectQuantity','addContinuousShowChoices','convertMessageCoreEscapeActions','convertLockColorsEscapeCharacters','bADGB','defaultColor','battle\x20actor','isItem','faceName','createContents','qHBDx','addedHeight','_cancelButton','changeValue','outLineColor','name','exec','boxHeight','KEeYo','innerWidth','prepareAutoSizeEscapeCharacters','FNQMU','paintOpacity','isWordWrapEnabled','isBusy','convertTextMacros','\x1bC[%1]%2\x1bPREVCOLOR[0]','updateNameBoxMove','pBHNa','map\x20player','itemPadding','textSizeEx','_textAlignment','addCommand','setChoiceListLineHeight','Classes','RqseG','isTriggered','map','processEscapeCharacter','convertBaseEscapeCharacters','_positionType','prepareShowTextFollowups','inputtingAction','mNIpG','nNDGz','addMessageCoreCommands','mainFontFace','version','parse','processStoredAutoColorChanges','_nameBoxWindow','_targets','TkZGt','resetRect','choiceCols','StretchDimmedBg','filter','placeCancelButton','textSpeed','windowPadding','description','ozFFB','left','[0]','prepareShowTextCommand','55554mPEIKk','CENTERPICTURE','yeXhq','easeIn','getLastGainedItemData','AutoColorBypassList','applyData','clamp','choice','code','isCommandEnabled','bijey','TvGhE','makeFontSmaller','Armors','setWordWrap','right','processAllText','ZPvqI','Window_Message_synchronizeNameBox','process_VisuMZ_MessageCore_TextCodes_Replace','messageWidth','TextAlign','processCharacter','isRTL','processAutoColorWords','Undefined','splice','resetTextColor','BOLD','emerge','shift','addExtraShowChoices','getMessageWindowWidth','lvLGi','CreateAutoColorFor','updateForcedPlacement','Window_Base_changeTextColor','terminateMessage','cwrNV','VbSUq','updateOverlappingY','ARRAYSTRUCT','VcDlE','Default','addLoadListener','getMessageWindowRows','setChoiceListMaxRows','kkhYt','preemptive','AutoColorRegExp','States','prototype','MessageCore','loadPicture','clear','open','faceWidth','NameBoxWindowOffsetY','textColor','commandSymbol','FUNC','_textDelayCount','</I>','changeTextSpeed','UvwvA','_index','onNewPageMessageCore','updateOffsetPosition','xQvRx','ActionJS','setMessageWindowWidth','choiceLineHeight','event','TightWrap','initTextAlignement','push','easeInOut','startX','type','convertVariableEscapeCharacters','Window_NameBox_updatePlacement','Window_Message_isTriggered','TAtSh','changePaintOpacity','Window_Base_initialize','drawing','\x1bCOLORLOCK[0]','processFsTextCode','<%1>','</COLORLOCK>','gOhnr','HvBQp','_spriteset','Enemies','fontSize','setupChoices','_autoSizeRegexp','ptHzn','_moveTargetY','SWITCHES','\x5c%1','NUM','makeCommandList','isInputting','convertMessageCoreEscapeReplacements','messageCoreTextSpeed','status','setMessageWindowRows','obtainItem','EndPadding','addMessageCoreTextSpeedCommand','AutoColor','PpqzM','slice','yJXjt','zeEgd','trim','WordWrap','findTargetSprite','3793820KVCRFk','victory','TextManager_message','FontChangeValue','arJRp','259SNHyVA','<B>','test','Actors','levelUp','messageWordWrap','follower','fontFace','Skills','format','Window_Options_changeVolume','\x1bWrapBreak[0]','center','stretchDimmerSprite','contentsBack','outlineColor','updatePlacement','changeVolume','convertShowChoiceEscapeCodes','fxnep','UYvYx','updateEvents','_centerMessageWindow','ChoiceWindowProperties','SAqha','battleUserName','processFontChangeItalic','Window_ChoiceList_updatePlacement','windowWidth','processMessageCoreEscapeActions','ARRAYNUM','_MessageCoreSettings','getChoiceListTextAlign','toLowerCase','textSizeExTextAlignment','Type','_action','callOkHandler','SHOW','ARRAYFUNC','command101','ChoiceWindowMaxCols','MvoBQ','index','isPressed','XozoH','Name','SortObjectByKeyLength','fUzQX','processDrawCenteredPicture','IfwPw','\x1bTEXTALIGNMENT[3]','prepareWordWrapEscapeCharacters','messageCoreWindowX','quantity','DISABLE','isChoiceVisible','Rows','clearCommandList','ParseClassNotetags','Items','join','ENABLE','1550728NYqeEG','10228293nHvWWI','setTextDelay','ConvertTextAutoColorRegExpFriendly','default','close','contents','replace','mIAQB','ITALIC','OiKMh','Window_Options_addGeneralOptions','drawBackCenteredPicture','choiceRows','_indent','VisuMZ_0_CoreEngine','Window_Message_processEscapeCharacter','getChoiceListMaxRows','resetFontSettings','</WORDWRAP>','currentExt','Game_System_initialize','setPositionType','AddOption','databaseObjectName','followers','FJOEB','numVisibleRows','STRUCT','updateBackground','cUqBF','Window_Options_isVolumeSymbol','dPwZg','setupItemChoice','addMessageCommonEvent','launchMessageCommonEvent','<BR>','match','\x1bITALIC[0]','clampPlacementPosition','calcMoveEasing','itemRectWithPadding','synchronizeNameBox','%1\x27s\x20version\x20does\x20not\x20match\x20plugin\x27s.\x20Please\x20update\x20it\x20in\x20the\x20Plugin\x20Manager.','convertEscapeCharacters','true','FontSmallerCap','ParseSkillNotetags','ARRAYSTR','Weapons','bind','convertHardcodedEscapeReplacements','processAutoSize','statusText','innerHeight','TextStr','isArmor','dMziv','%1\x20is\x20missing\x20a\x20required\x20plugin.\x0aPlease\x20install\x20%2\x20into\x20the\x20Plugin\x20Manager.','_relativePosition','list','ChoiceWindowLineHeight','OluAV','messageRows','Window_ChoiceList_windowX','_moveDuration','_moveTargetHeight','Instant','processPxTextCode','CreateAutoColorRegExpLists','Scene_Boot_onDatabaseLoaded','adjustShowChoiceDefault','Game_Party_initialize','LineBreakSpace','Window_Base_processEscapeCharacter','contentsHeight','TextSpeed','preFlushTextState','PICTURE','\x1bCOLORLOCK[1]','updateDimensions','dfRxA','lineHeight','_wholeMoveDuration','outlineWidth','RelativePXPY','isAutoColorAffected','makeDeepCopy','textSizeExWordWrap','_interpreter','MessageWindow','processFontChangeBold','substr','_moveEasingType','setRelativePosition','parameters','onProcessCharacter','rtl','maxCommands','add','clearFlags','makeFontBigger','processAutoPosition','text','members','Settings','Game_Map_initialize','Game_Map_setupEvents','openness','anchor','indent','_wordWrap','Center','CommonEvent','choicePositionType','maxCols','processCommonEvent','process_VisuMZ_MessageCore_TextCodes_Action','fontItalic','width','setFaceImage','mDdoT','isColorLocked','setChoiceListTextAlign','ZiHGj','\x1bI[%1]','indexOf','setMessageWindowWordWrap','YdxFX','commandName','_commonEventId','ceil','Window_NameBox_refresh','Window_Base_update','RzxWO','constructor','Window_Message_newPage','10DXFBdE','isSceneMap','helpWordWrap','Game_Interpreter_setupChoices','updateMessageCommonEvents','_autoPositionTarget','rOkLs','nGHXh','onChoice','Game_Map_updateEvents','updateAutoPosition','_scene','includes','_autoPosRegExp','<LINE\x20BREAK>','RLeOt','none','maxFontSizeInLine','setChoiceListMaxColumns','dzlXf'];_0x1363=function(){return _0x69b930;};return _0x1363();}VisuMZ[label]['Settings']=VisuMZ[label][_0x22ff2b(0x236)]||{},VisuMZ[_0x22ff2b(0x277)]=function(_0x3a4acb,_0x4cf91f){const _0x1834e8=_0x22ff2b;for(const _0x561cbe in _0x4cf91f){if(_0x561cbe[_0x1834e8(0x1f2)](/(.*):(.*)/i)){if(_0x1834e8(0x31c)!=='YqJdw'){const _0x2ef18f=String(RegExp['$1']),_0x358525=String(RegExp['$2'])[_0x1834e8(0x2a4)]()[_0x1834e8(0x415)]();let _0x5eceac,_0x3e7d61,_0x4ece7d;switch(_0x358525){case _0x1834e8(0x406):_0x5eceac=_0x4cf91f[_0x561cbe]!==''?Number(_0x4cf91f[_0x561cbe]):0x0;break;case _0x1834e8(0x1ac):_0x3e7d61=_0x4cf91f[_0x561cbe]!==''?JSON[_0x1834e8(0x38f)](_0x4cf91f[_0x561cbe]):[],_0x5eceac=_0x3e7d61['map'](_0x2d8db9=>Number(_0x2d8db9));break;case _0x1834e8(0x340):_0x5eceac=_0x4cf91f[_0x561cbe]!==''?eval(_0x4cf91f[_0x561cbe]):null;break;case _0x1834e8(0x345):_0x3e7d61=_0x4cf91f[_0x561cbe]!==''?JSON[_0x1834e8(0x38f)](_0x4cf91f[_0x561cbe]):[],_0x5eceac=_0x3e7d61[_0x1834e8(0x384)](_0x18cba9=>eval(_0x18cba9));break;case _0x1834e8(0x2b8):_0x5eceac=_0x4cf91f[_0x561cbe]!==''?JSON['parse'](_0x4cf91f[_0x561cbe]):'';break;case _0x1834e8(0x33d):_0x3e7d61=_0x4cf91f[_0x561cbe]!==''?JSON[_0x1834e8(0x38f)](_0x4cf91f[_0x561cbe]):[],_0x5eceac=_0x3e7d61[_0x1834e8(0x384)](_0x443732=>JSON['parse'](_0x443732));break;case _0x1834e8(0x3dd):_0x5eceac=_0x4cf91f[_0x561cbe]!==''?new Function(JSON[_0x1834e8(0x38f)](_0x4cf91f[_0x561cbe])):new Function('return\x200');break;case _0x1834e8(0x1b5):_0x3e7d61=_0x4cf91f[_0x561cbe]!==''?JSON[_0x1834e8(0x38f)](_0x4cf91f[_0x561cbe]):[],_0x5eceac=_0x3e7d61[_0x1834e8(0x384)](_0x404dfc=>new Function(JSON[_0x1834e8(0x38f)](_0x404dfc)));break;case'STR':_0x5eceac=_0x4cf91f[_0x561cbe]!==''?String(_0x4cf91f[_0x561cbe]):'';break;case _0x1834e8(0x1fd):_0x3e7d61=_0x4cf91f[_0x561cbe]!==''?JSON[_0x1834e8(0x38f)](_0x4cf91f[_0x561cbe]):[],_0x5eceac=_0x3e7d61[_0x1834e8(0x384)](_0x179ec2=>String(_0x179ec2));break;case _0x1834e8(0x1e9):_0x4ece7d=_0x4cf91f[_0x561cbe]!==''?JSON[_0x1834e8(0x38f)](_0x4cf91f[_0x561cbe]):{},_0x3a4acb[_0x2ef18f]={},VisuMZ[_0x1834e8(0x277)](_0x3a4acb[_0x2ef18f],_0x4ece7d);continue;case _0x1834e8(0x3ca):_0x3e7d61=_0x4cf91f[_0x561cbe]!==''?JSON[_0x1834e8(0x38f)](_0x4cf91f[_0x561cbe]):[],_0x5eceac=_0x3e7d61[_0x1834e8(0x384)](_0x76d47d=>VisuMZ[_0x1834e8(0x277)]({},JSON[_0x1834e8(0x38f)](_0x76d47d)));break;default:continue;}_0x3a4acb[_0x2ef18f]=_0x5eceac;}else this[_0x1834e8(0x3af)](![]);}}return _0x3a4acb;},(_0x370e6a=>{const _0x506e1a=_0x22ff2b,_0x4f4eca=_0x370e6a[_0x506e1a(0x36d)];for(const _0x3711b7 of dependencies){if(_0x506e1a(0x28d)===_0x506e1a(0x28d)){if(!Imported[_0x3711b7]){alert(_0x506e1a(0x207)[_0x506e1a(0x197)](_0x4f4eca,_0x3711b7)),SceneManager['exit']();break;}}else{this[_0x506e1a(0x2c2)]=this[_0x506e1a(0x2c2)]||[];for(const _0x53915c of this[_0x506e1a(0x2c2)]){!_0x53915c['_interpreter']?this[_0x506e1a(0x2c2)][_0x506e1a(0x28a)](_0x53915c):_0x53915c['update']();}}}const _0x1efd3f=_0x370e6a[_0x506e1a(0x39b)];if(_0x1efd3f[_0x506e1a(0x1f2)](/\[Version[ ](.*?)\]/i)){const _0x50ce2a=Number(RegExp['$1']);_0x50ce2a!==VisuMZ[label][_0x506e1a(0x38e)]&&(alert(_0x506e1a(0x1f8)['format'](_0x4f4eca,_0x50ce2a)),SceneManager[_0x506e1a(0x341)]());}if(_0x1efd3f[_0x506e1a(0x1f2)](/\[Tier[ ](\d+)\]/i)){if(_0x506e1a(0x2ae)!==_0x506e1a(0x2d7)){const _0x2a4a0a=Number(RegExp['$1']);if(_0x2a4a0a<tier)'TtMha'!=='GNEJt'?(alert('%1\x20is\x20incorrectly\x20placed\x20on\x20the\x20plugin\x20list.\x0aIt\x20is\x20a\x20Tier\x20%2\x20plugin\x20placed\x20over\x20other\x20Tier\x20%3\x20plugins.\x0aPlease\x20reorder\x20the\x20plugin\x20list\x20from\x20smallest\x20to\x20largest\x20tier\x20numbers.'['format'](_0x4f4eca,_0x2a4a0a,tier)),SceneManager[_0x506e1a(0x341)]()):(this[_0x506e1a(0x226)]=new _0x5140be(),this[_0x506e1a(0x226)][_0x506e1a(0x281)](this[_0x506e1a(0x209)](),this[_0x506e1a(0x352)]));else{if('pBHNa'!==_0x506e1a(0x37a))return 0x0;else tier=Math[_0x506e1a(0x2d3)](_0x2a4a0a,tier);}}else return _0x506e1a(0x378)[_0x506e1a(0x197)](_0x381c07,_0x3b7a63);}VisuMZ['ConvertParams'](VisuMZ[label][_0x506e1a(0x236)],_0x370e6a[_0x506e1a(0x22c)]);})(pluginData),PluginManager[_0x22ff2b(0x2df)](pluginData[_0x22ff2b(0x36d)],_0x22ff2b(0x1a5),_0x5988ce=>{const _0x3288a8=_0x22ff2b;VisuMZ[_0x3288a8(0x277)](_0x5988ce,_0x5988ce);const _0x28e314=_0x5988ce['LineHeight']||$gameSystem['getChoiceListLineHeight']()||0x1,_0x32006d=_0x5988ce[_0x3288a8(0x2b6)]||$gameSystem[_0x3288a8(0x1de)]()||0x1,_0x183a1f=_0x5988ce[_0x3288a8(0x356)]||$gameSystem[_0x3288a8(0x2a6)]()||0x1,_0xc818c7=_0x5988ce[_0x3288a8(0x3b6)]['toLowerCase']()||'default';$gameSystem['setChoiceListLineHeight'](_0x28e314),$gameSystem['setChoiceListMaxRows'](_0x32006d),$gameSystem[_0x3288a8(0x268)](_0x183a1f),$gameSystem[_0x3288a8(0x248)](_0xc818c7);}),PluginManager[_0x22ff2b(0x2df)](pluginData['name'],'MessageWindowProperties',_0x5c991b=>{const _0x3549b7=_0x22ff2b;VisuMZ['ConvertParams'](_0x5c991b,_0x5c991b);const _0x155059=_0x5c991b[_0x3549b7(0x1c7)]||$gameSystem[_0x3549b7(0x3ce)]()||0x1,_0x239dbf=_0x5c991b[_0x3549b7(0x289)]||$gameSystem['getMessageWindowWidth']()||0x1;$gameTemp['_centerMessageWindow']=_0x5c991b[_0x3549b7(0x23d)]||![];const _0x551aa5=_0x5c991b['WordWrap'][_0x3549b7(0x1af)]();$gameSystem['setMessageWindowRows'](_0x155059),$gameSystem['setMessageWindowWidth'](_0x239dbf);[_0x3549b7(0x1fa),'false'][_0x3549b7(0x262)](_0x551aa5)&&$gameSystem[_0x3549b7(0x24c)](eval(_0x551aa5));const _0x430634=SceneManager[_0x3549b7(0x261)][_0x3549b7(0x2a5)];_0x430634&&(_0x430634[_0x3549b7(0x33b)](),_0x430634[_0x3549b7(0x21d)](),_0x430634['createContents']());}),VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x213)]=Scene_Boot[_0x22ff2b(0x3d4)][_0x22ff2b(0x327)],Scene_Boot[_0x22ff2b(0x3d4)][_0x22ff2b(0x327)]=function(){const _0x2e3a39=_0x22ff2b;VisuMZ[_0x2e3a39(0x3d5)][_0x2e3a39(0x213)][_0x2e3a39(0x282)](this),this[_0x2e3a39(0x242)](),this[_0x2e3a39(0x3b4)](),this[_0x2e3a39(0x30c)](),this['process_VisuMZ_MessageCore_AutoColor']();},VisuMZ[_0x22ff2b(0x3d5)]['SortObjectByKeyLength']=function(_0x2106a9){const _0x328cbe=_0x22ff2b,_0x3556b0=VisuMZ['MessageCore'][_0x328cbe(0x236)][_0x2106a9];_0x3556b0[_0x328cbe(0x335)]((_0x45376c,_0x327240)=>{const _0x46db34=_0x328cbe;if(_0x46db34(0x1d7)!=='KCTBX'){if(!_0x45376c||!_0x327240)return-0x1;return _0x327240[_0x46db34(0x27e)][_0x46db34(0x2fe)]-_0x45376c[_0x46db34(0x27e)][_0x46db34(0x2fe)];}else{if(this[_0x46db34(0x1ad)]===_0x17753a)this[_0x46db34(0x35d)]();if(this[_0x46db34(0x1ad)][_0x46db34(0x20c)]===_0x74cf5f)this[_0x46db34(0x35d)]();this[_0x46db34(0x1ad)]['messageRows']=_0x1d5d9d||0x1;}});},Scene_Boot['prototype'][_0x22ff2b(0x242)]=function(){const _0x558c91=_0x22ff2b;VisuMZ[_0x558c91(0x3d5)][_0x558c91(0x1bd)](_0x558c91(0x2ec));for(const _0x985b37 of VisuMZ['MessageCore']['Settings'][_0x558c91(0x2ec)]){_0x985b37['Match']=_0x985b37[_0x558c91(0x27e)]['toUpperCase'](),_0x985b37[_0x558c91(0x2ea)]=new RegExp('\x1b'+_0x985b37[_0x558c91(0x27e)],'gi'),_0x985b37[_0x558c91(0x323)]='\x1b'+_0x985b37[_0x558c91(0x27e)];if(_0x985b37[_0x558c91(0x1b1)]==='')_0x985b37[_0x558c91(0x323)]+=_0x558c91(0x39e);}},Scene_Boot['prototype'][_0x22ff2b(0x3b4)]=function(){const _0x3d0040=_0x22ff2b;VisuMZ[_0x3d0040(0x3d5)]['SortObjectByKeyLength']('TextCodeReplace');for(const _0x1399d2 of VisuMZ[_0x3d0040(0x3d5)][_0x3d0040(0x236)][_0x3d0040(0x2a9)]){_0x1399d2[_0x3d0040(0x2ea)]=new RegExp('\x1b'+_0x1399d2[_0x3d0040(0x27e)]+_0x1399d2[_0x3d0040(0x1b1)],'gi');if(_0x1399d2[_0x3d0040(0x204)]!==''&&_0x1399d2[_0x3d0040(0x204)]!==_0x3d0040(0x3ba))_0x1399d2[_0x3d0040(0x323)]=new Function(_0x3d0040(0x326)+_0x1399d2[_0x3d0040(0x204)][_0x3d0040(0x1d4)](/\\/g,'\x1b')+'\x27');else{if(_0x3d0040(0x1e7)!==_0x3d0040(0x319))_0x1399d2[_0x3d0040(0x323)]=_0x1399d2[_0x3d0040(0x338)];else{const _0x30a25d=_0x25b697[_0x3d0040(0x3d5)][_0x3d0040(0x236)][_0x3d0040(0x410)];!_0x1eaae1[_0x3d0040(0x2f6)]&&(_0x3d678a['MessageCore'][_0x3d0040(0x348)](_0x24ac2e,_0x30a25d[_0x3d0040(0x381)]),_0x50803e['MessageCore'][_0x3d0040(0x348)](_0x270c54,_0x30a25d['Skills']),_0x4741c9[_0x3d0040(0x3d5)][_0x3d0040(0x348)](_0x405185,_0x30a25d[_0x3d0040(0x1ca)]),_0x489cb9[_0x3d0040(0x3d5)][_0x3d0040(0x348)](_0x242993,_0x30a25d[_0x3d0040(0x1fe)]),_0xaaf333[_0x3d0040(0x3d5)]['AddAutoColor'](_0x2b32c1,_0x30a25d['Armors']),_0xccb88e[_0x3d0040(0x3d5)]['AddAutoColor'](_0x3ad5ec,_0x30a25d[_0x3d0040(0x3fe)]),_0x1a50bc['MessageCore'][_0x3d0040(0x348)](_0x59957f,_0x30a25d[_0x3d0040(0x3d3)])),_0x34e50c[_0x3d0040(0x3d5)][_0x3d0040(0x212)]();}}}},Scene_Boot[_0x22ff2b(0x3d4)]['process_VisuMZ_MessageCore_TextMacros']=function(){const _0x184d83=_0x22ff2b;for(const _0x3f6ed9 of VisuMZ[_0x184d83(0x3d5)]['Settings'][_0x184d83(0x295)]){if(_0x184d83(0x1a2)===_0x184d83(0x1a2))_0x3f6ed9['textCodeCheck']=new RegExp('\x5c['+_0x3f6ed9['Match']+'\x5c]','gi'),_0x3f6ed9[_0x184d83(0x204)]!==''&&_0x3f6ed9['TextStr']!=='Undefined'?_0x3f6ed9['textCodeResult']=new Function(_0x184d83(0x326)+_0x3f6ed9['TextStr'][_0x184d83(0x1d4)](/\\/g,'\x1b')+'\x27'):_0x3f6ed9[_0x184d83(0x323)]=_0x3f6ed9[_0x184d83(0x338)];else{if(_0x4d481e['Type']==='')this[_0x184d83(0x2ab)](_0x2340af);_0x1e8bc7[_0x184d83(0x3e6)]['call'](this,_0x4af9a9);if(this['constructor']===_0x278921){const _0xcd91db=_0x3f3808[_0x184d83(0x23e)]||0x0;if(_0xcd91db>0x0)this[_0x184d83(0x1f0)](_0xcd91db);}}}},Scene_Boot[_0x22ff2b(0x3d4)][_0x22ff2b(0x2c7)]=function(){const _0xa5ab18=_0x22ff2b,_0x5b3a96=VisuMZ[_0xa5ab18(0x3d5)][_0xa5ab18(0x236)][_0xa5ab18(0x410)];!VisuMZ['ParseAllNotetags']&&(VisuMZ[_0xa5ab18(0x3d5)][_0xa5ab18(0x348)]($dataClasses,_0x5b3a96['Classes']),VisuMZ['MessageCore'][_0xa5ab18(0x348)]($dataSkills,_0x5b3a96[_0xa5ab18(0x196)]),VisuMZ[_0xa5ab18(0x3d5)][_0xa5ab18(0x348)]($dataItems,_0x5b3a96['Items']),VisuMZ['MessageCore'][_0xa5ab18(0x348)]($dataWeapons,_0x5b3a96['Weapons']),VisuMZ['MessageCore'][_0xa5ab18(0x348)]($dataArmors,_0x5b3a96[_0xa5ab18(0x3ae)]),VisuMZ[_0xa5ab18(0x3d5)]['AddAutoColor']($dataEnemies,_0x5b3a96[_0xa5ab18(0x3fe)]),VisuMZ['MessageCore'][_0xa5ab18(0x348)]($dataStates,_0x5b3a96[_0xa5ab18(0x3d3)])),VisuMZ[_0xa5ab18(0x3d5)][_0xa5ab18(0x212)]();},VisuMZ[_0x22ff2b(0x3d5)]['AutoColorBypassList']=['V','N','P','C','I','PX','PY','G','{','}','<','>','FS','\x5c','$','.','|','!','<','>','^',_0x22ff2b(0x41e),_0x22ff2b(0x314),_0x22ff2b(0x349),_0x22ff2b(0x3df),_0x22ff2b(0x35b),_0x22ff2b(0x27a),'<CENTER>','</CENTER>',_0x22ff2b(0x27c),_0x22ff2b(0x322),'<COLORLOCK>',_0x22ff2b(0x3fa),'(((',_0x22ff2b(0x291),'<WORDWRAP>',_0x22ff2b(0x1e0),_0x22ff2b(0x1f1),_0x22ff2b(0x264),'PICTURE',_0x22ff2b(0x3a1),'COMMONEVENT',_0x22ff2b(0x343),_0x22ff2b(0x1b4),'HIDE',_0x22ff2b(0x1cc),_0x22ff2b(0x1c5),'SWITCH',_0x22ff2b(0x404),'ALL','ANY'],VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x348)]=function(_0x4ea0c1,_0x5042f9){const _0x985e56=_0x22ff2b;if(_0x5042f9<=0x0)return;const _0x2b4f84=_0x4ea0c1;for(const _0x8af65e of _0x2b4f84){if(!_0x8af65e)continue;VisuMZ[_0x985e56(0x3d5)][_0x985e56(0x3c3)](_0x8af65e,_0x5042f9);}},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x212)]=function(){const _0x394dd6=_0x22ff2b;VisuMZ['MessageCore']['AutoColorRegExp']=[];for(let _0x14ab43=0x1;_0x14ab43<=0x1f;_0x14ab43++){if(_0x394dd6(0x21e)===_0x394dd6(0x2b3))this[_0x394dd6(0x2dc)]()&&(this['x']=this[_0x394dd6(0x284)](this['x'],this[_0x394dd6(0x2a0)]),this['y']=this['applyMoveEasing'](this['y'],this['_moveTargetY']),this[_0x394dd6(0x244)]=this['applyMoveEasing'](this[_0x394dd6(0x244)],this['_moveTargetWidth']),this[_0x394dd6(0x293)]=this[_0x394dd6(0x284)](this[_0x394dd6(0x293)],this['_moveTargetHeight']),this[_0x394dd6(0x1f4)]()),this[_0x394dd6(0x20e)]--;else{const _0x12d7cd='TextColor%1'[_0x394dd6(0x197)](_0x14ab43),_0xc80099=VisuMZ[_0x394dd6(0x3d5)][_0x394dd6(0x236)]['AutoColor'][_0x12d7cd];_0xc80099[_0x394dd6(0x335)]((_0x1932b2,_0x5db04b)=>{const _0x4160ef=_0x394dd6;if(!_0x1932b2||!_0x5db04b)return-0x1;return _0x5db04b[_0x4160ef(0x2fe)]-_0x1932b2['length'];}),this[_0x394dd6(0x279)](_0xc80099,_0x14ab43);}}},VisuMZ['MessageCore']['CreateAutoColorRegExpListEntries']=function(_0x447cc,_0x3c160e){const _0x572b3a=_0x22ff2b;for(const _0x952b4d of _0x447cc){if(_0x952b4d[_0x572b3a(0x2fe)]<=0x0)continue;if(/^\d+$/[_0x572b3a(0x41f)](_0x952b4d))continue;let _0x5e8d97=VisuMZ[_0x572b3a(0x3d5)]['ConvertTextAutoColorRegExpFriendly'](_0x952b4d);if(_0x952b4d[_0x572b3a(0x1f2)](/[\u3000-\u303F]|[\u3040-\u309F]|[\u30A0-\u30FF]|[\uFF00-\uFFEF]|[\u4E00-\u9FAF]|[\u2605-\u2606]|[\u2190-\u2195]|\u203B/g)){if(_0x572b3a(0x3c8)===_0x572b3a(0x320)){_0x20fdbb[_0x572b3a(0x27e)]=_0x48168a[_0x572b3a(0x27e)][_0x572b3a(0x2a4)](),_0x3af4f8[_0x572b3a(0x2ea)]=new _0x43ccf6('\x1b'+_0x3a1cf3[_0x572b3a(0x27e)],'gi'),_0x1ff283[_0x572b3a(0x323)]='\x1b'+_0x14ac7f[_0x572b3a(0x27e)];if(_0xaf12dc[_0x572b3a(0x1b1)]==='')_0x5752ea[_0x572b3a(0x323)]+=_0x572b3a(0x39e);}else var _0x20083e=new RegExp(_0x5e8d97,'i');}else var _0x20083e=new RegExp('\x5cb'+_0x5e8d97+'\x5cb','g');VisuMZ[_0x572b3a(0x3d5)][_0x572b3a(0x3d2)]['push']([_0x20083e,'\x1bC[%1]%2\x1bPREVCOLOR[0]'[_0x572b3a(0x197)](_0x3c160e,_0x952b4d)]);}},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x1d0)]=function(_0x58ab86){const _0x1ce5d4=_0x22ff2b;return _0x58ab86=_0x58ab86[_0x1ce5d4(0x1d4)](/(\W)/gi,(_0x571627,_0x4958cb)=>_0x1ce5d4(0x405)[_0x1ce5d4(0x197)](_0x4958cb)),_0x58ab86;},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x1c9)]=VisuMZ[_0x22ff2b(0x1c9)],VisuMZ[_0x22ff2b(0x1c9)]=function(_0x28c32f){const _0x1e9b9f=_0x22ff2b;VisuMZ[_0x1e9b9f(0x3d5)][_0x1e9b9f(0x1c9)][_0x1e9b9f(0x282)](this,_0x28c32f);const _0x409e10=VisuMZ[_0x1e9b9f(0x3d5)][_0x1e9b9f(0x236)][_0x1e9b9f(0x410)];VisuMZ[_0x1e9b9f(0x3d5)][_0x1e9b9f(0x3c3)](_0x28c32f,_0x409e10[_0x1e9b9f(0x381)]);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x1fc)]=VisuMZ[_0x22ff2b(0x1fc)],VisuMZ[_0x22ff2b(0x1fc)]=function(_0x1bc22c){const _0x442af3=_0x22ff2b;VisuMZ[_0x442af3(0x3d5)][_0x442af3(0x1fc)][_0x442af3(0x282)](this,_0x1bc22c);const _0x462653=VisuMZ[_0x442af3(0x3d5)][_0x442af3(0x236)][_0x442af3(0x410)];VisuMZ[_0x442af3(0x3d5)]['CreateAutoColorFor'](_0x1bc22c,_0x462653[_0x442af3(0x196)]);},VisuMZ[_0x22ff2b(0x3d5)]['ParseItemNotetags']=VisuMZ[_0x22ff2b(0x2eb)],VisuMZ['ParseItemNotetags']=function(_0x321964){const _0x1f9c4c=_0x22ff2b;VisuMZ[_0x1f9c4c(0x3d5)][_0x1f9c4c(0x2eb)][_0x1f9c4c(0x282)](this,_0x321964);const _0x3d8bf4=VisuMZ[_0x1f9c4c(0x3d5)][_0x1f9c4c(0x236)][_0x1f9c4c(0x410)];VisuMZ['MessageCore']['CreateAutoColorFor'](_0x321964,_0x3d8bf4[_0x1f9c4c(0x1ca)]);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x305)]=VisuMZ[_0x22ff2b(0x305)],VisuMZ[_0x22ff2b(0x305)]=function(_0x15350f){const _0x387013=_0x22ff2b;VisuMZ[_0x387013(0x3d5)]['ParseWeaponNotetags'][_0x387013(0x282)](this,_0x15350f);const _0x1fc680=VisuMZ[_0x387013(0x3d5)][_0x387013(0x236)][_0x387013(0x410)];VisuMZ[_0x387013(0x3d5)]['CreateAutoColorFor'](_0x15350f,_0x1fc680[_0x387013(0x1fe)]);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x287)]=VisuMZ['ParseArmorNotetags'],VisuMZ[_0x22ff2b(0x287)]=function(_0x31b192){const _0x3353f9=_0x22ff2b;VisuMZ[_0x3353f9(0x3d5)][_0x3353f9(0x287)][_0x3353f9(0x282)](this,_0x31b192);const _0x470cdc=VisuMZ[_0x3353f9(0x3d5)][_0x3353f9(0x236)][_0x3353f9(0x410)];VisuMZ[_0x3353f9(0x3d5)][_0x3353f9(0x3c3)](_0x31b192,_0x470cdc['Armors']);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x276)]=VisuMZ['ParseEnemyNotetags'],VisuMZ['ParseEnemyNotetags']=function(_0x35247d){const _0x533d13=_0x22ff2b;VisuMZ[_0x533d13(0x3d5)][_0x533d13(0x276)]['call'](this,_0x35247d);const _0x4320e4=VisuMZ['MessageCore'][_0x533d13(0x236)][_0x533d13(0x410)];VisuMZ['MessageCore'][_0x533d13(0x3c3)](_0x35247d,_0x4320e4[_0x533d13(0x3fe)]);},VisuMZ['MessageCore'][_0x22ff2b(0x33c)]=VisuMZ[_0x22ff2b(0x33c)],VisuMZ[_0x22ff2b(0x33c)]=function(_0x1fcd4a){const _0x617cca=_0x22ff2b;VisuMZ[_0x617cca(0x3d5)][_0x617cca(0x33c)][_0x617cca(0x282)](this,_0x1fcd4a);const _0x50aec6=VisuMZ[_0x617cca(0x3d5)]['Settings']['AutoColor'];VisuMZ['MessageCore'][_0x617cca(0x3c3)](_0x1fcd4a,_0x50aec6[_0x617cca(0x3d3)]);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x3c3)]=function(_0x3e373f,_0x470a06){const _0x5ea7db=_0x22ff2b;if(_0x470a06<=0x0)return;const _0x44d74a=VisuMZ[_0x5ea7db(0x3d5)][_0x5ea7db(0x236)][_0x5ea7db(0x410)]['TextColor'+_0x470a06];let _0xc2f02a=_0x3e373f['name'][_0x5ea7db(0x415)]();if(/^\d+$/[_0x5ea7db(0x41f)](_0xc2f02a))return;if(VisuMZ['MessageCore'][_0x5ea7db(0x3a5)][_0x5ea7db(0x262)](_0xc2f02a[_0x5ea7db(0x2a4)]()))return;_0xc2f02a=_0xc2f02a[_0x5ea7db(0x1d4)](/\\I\[(\d+)\]/gi,''),_0xc2f02a=_0xc2f02a[_0x5ea7db(0x1d4)](/\x1bI\[(\d+)\]/gi,'');if(_0xc2f02a[_0x5ea7db(0x2fe)]<=0x0)return;if(_0xc2f02a[_0x5ea7db(0x1f2)](/-----/i))return;_0x44d74a[_0x5ea7db(0x3ec)](_0xc2f02a);},SceneManager[_0x22ff2b(0x35a)]=function(){const _0x289f0d=_0x22ff2b;return this[_0x289f0d(0x261)]&&this[_0x289f0d(0x261)][_0x289f0d(0x254)]===Scene_Battle;},SceneManager[_0x22ff2b(0x257)]=function(){const _0x3c5e40=_0x22ff2b;return this[_0x3c5e40(0x261)]&&this[_0x3c5e40(0x261)][_0x3c5e40(0x254)]===Scene_Map;},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x41a)]=TextManager[_0x22ff2b(0x328)],TextManager[_0x22ff2b(0x328)]=function(_0x3526e9){const _0x46ed68=_0x22ff2b,_0x2637f5=[_0x46ed68(0x421),_0x46ed68(0x3be),_0x46ed68(0x3d1),_0x46ed68(0x31e),_0x46ed68(0x419),_0x46ed68(0x31d),'escapeStart',_0x46ed68(0x33a),'obtainGold',_0x46ed68(0x40d)];let _0x2bc03f=VisuMZ[_0x46ed68(0x3d5)][_0x46ed68(0x41a)]['call'](this,_0x3526e9);return _0x2637f5[_0x46ed68(0x262)](_0x3526e9)&&(_0x2bc03f=_0x46ed68(0x1e0)+_0x2bc03f),_0x2bc03f;},ConfigManager[_0x22ff2b(0x399)]=VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x236)][_0x22ff2b(0x219)]['Default'],VisuMZ[_0x22ff2b(0x3d5)]['ConfigManager_makeData']=ConfigManager['makeData'],ConfigManager[_0x22ff2b(0x311)]=function(){const _0x1bc5c1=_0x22ff2b,_0x3ff0a9=VisuMZ[_0x1bc5c1(0x3d5)]['ConfigManager_makeData'][_0x1bc5c1(0x282)](this);return _0x3ff0a9[_0x1bc5c1(0x399)]=this[_0x1bc5c1(0x399)],_0x3ff0a9;},VisuMZ['MessageCore'][_0x22ff2b(0x2b7)]=ConfigManager[_0x22ff2b(0x3a6)],ConfigManager[_0x22ff2b(0x3a6)]=function(_0x3e70ac){const _0x24a410=_0x22ff2b;VisuMZ['MessageCore']['ConfigManager_applyData'][_0x24a410(0x282)](this,_0x3e70ac);if(_0x24a410(0x399)in _0x3e70ac)this[_0x24a410(0x399)]=Number(_0x3e70ac['textSpeed'])[_0x24a410(0x3a7)](0x1,0xb);else{if(_0x24a410(0x275)===_0x24a410(0x275))this['textSpeed']=VisuMZ[_0x24a410(0x3d5)]['Settings']['TextSpeed'][_0x24a410(0x3cc)];else{if(!this[_0x24a410(0x25b)])return;const _0x288dff=_0x82fa1c['_scene'];if(!_0x288dff)return;if(!_0x288dff[_0x24a410(0x3fd)])return;const _0x36b5bc=_0x288dff[_0x24a410(0x3fd)][_0x24a410(0x417)](this[_0x24a410(0x25b)]);if(!_0x36b5bc)return;let _0x46ecbd=_0x36b5bc['x'];_0x46ecbd-=this[_0x24a410(0x244)]/0x2,_0x46ecbd-=(_0xdf70fd[_0x24a410(0x244)]-_0x4b59b6[_0x24a410(0x26f)])/0x2;let _0xe02a1d=_0x36b5bc['y'];_0xe02a1d-=this[_0x24a410(0x293)],_0xe02a1d-=(_0x12b87a[_0x24a410(0x293)]-_0x58a829[_0x24a410(0x36f)])/0x2,_0xe02a1d-=_0x36b5bc[_0x24a410(0x293)]+0x8,this['x']=_0x758490[_0x24a410(0x278)](_0x46ecbd),this['y']=_0x3452ca[_0x24a410(0x278)](_0xe02a1d),this['clampPlacementPosition'](!![],![]),this[_0x24a410(0x391)]['updatePlacement']();}}},TextManager[_0x22ff2b(0x40a)]=VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x236)][_0x22ff2b(0x219)][_0x22ff2b(0x1bc)],TextManager['instantTextSpeed']=VisuMZ['MessageCore'][_0x22ff2b(0x236)][_0x22ff2b(0x219)][_0x22ff2b(0x210)],VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x1e2)]=Game_System[_0x22ff2b(0x3d4)]['initialize'],Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x2c1)]=function(){const _0x1dad55=_0x22ff2b;VisuMZ[_0x1dad55(0x3d5)][_0x1dad55(0x1e2)][_0x1dad55(0x282)](this),this[_0x1dad55(0x35d)]();},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x35d)]=function(){const _0x4f9207=_0x22ff2b,_0x41f417=VisuMZ[_0x4f9207(0x3d5)][_0x4f9207(0x236)][_0x4f9207(0x2da)],_0x59c6e1=VisuMZ[_0x4f9207(0x3d5)][_0x4f9207(0x236)][_0x4f9207(0x416)];this['_MessageCoreSettings']={'messageRows':_0x41f417['MessageRows'],'messageWidth':_0x41f417['MessageWidth'],'messageWordWrap':_0x59c6e1['MessageWindow'],'helpWordWrap':_0x59c6e1['HelpWindow'],'choiceLineHeight':_0x41f417['ChoiceWindowLineHeight'],'choiceRows':_0x41f417['ChoiceWindowMaxRows'],'choiceCols':_0x41f417[_0x4f9207(0x1b7)],'choiceTextAlign':_0x41f417['ChoiceWindowTextAlign']};},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x3ce)]=function(){const _0x2b93e5=_0x22ff2b;if(this[_0x2b93e5(0x1ad)]===undefined)this[_0x2b93e5(0x35d)]();if(this['_MessageCoreSettings'][_0x2b93e5(0x20c)]===undefined)this['initMessageCore']();return this[_0x2b93e5(0x1ad)]['messageRows'];},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x40c)]=function(_0x37eaeb){const _0x2a102f=_0x22ff2b;if(this[_0x2a102f(0x1ad)]===undefined)this[_0x2a102f(0x35d)]();if(this[_0x2a102f(0x1ad)][_0x2a102f(0x20c)]===undefined)this[_0x2a102f(0x35d)]();this['_MessageCoreSettings'][_0x2a102f(0x20c)]=_0x37eaeb||0x1;},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x3c1)]=function(){const _0x20f593=_0x22ff2b;if(this['_MessageCoreSettings']===undefined)this['initMessageCore']();if(this[_0x20f593(0x1ad)][_0x20f593(0x3b5)]===undefined)this['initMessageCore']();return this[_0x20f593(0x1ad)][_0x20f593(0x3b5)];},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x3e7)]=function(_0x4dbf60){const _0x1662ff=_0x22ff2b;if(this[_0x1662ff(0x1ad)]===undefined)this[_0x1662ff(0x35d)]();if(this[_0x1662ff(0x1ad)][_0x1662ff(0x3b5)]===undefined)this[_0x1662ff(0x35d)]();_0x4dbf60=Math[_0x1662ff(0x250)](_0x4dbf60);if(_0x4dbf60%0x2!==0x0)_0x4dbf60+=0x1;this['_MessageCoreSettings'][_0x1662ff(0x3b5)]=_0x4dbf60||0x2;},Game_System['prototype'][_0x22ff2b(0x272)]=function(){const _0x592b5e=_0x22ff2b;if(this[_0x592b5e(0x1ad)]===undefined)this['initMessageCore']();if(this[_0x592b5e(0x1ad)][_0x592b5e(0x422)]===undefined)this[_0x592b5e(0x35d)]();return this[_0x592b5e(0x1ad)]['messageWordWrap'];},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x24c)]=function(_0x2df2fa){const _0x430d4b=_0x22ff2b;if(this[_0x430d4b(0x1ad)]===undefined)this['initMessageCore']();if(this[_0x430d4b(0x1ad)]['messageWordWrap']===undefined)this[_0x430d4b(0x35d)]();this['_MessageCoreSettings'][_0x430d4b(0x422)]=_0x2df2fa;},Game_System[_0x22ff2b(0x3d4)]['isHelpWindowWordWrap']=function(){const _0x5bcda1=_0x22ff2b;if(this[_0x5bcda1(0x1ad)]===undefined)this['initMessageCore']();if(this[_0x5bcda1(0x1ad)]['helpWordWrap']===undefined)this[_0x5bcda1(0x35d)]();return this[_0x5bcda1(0x1ad)][_0x5bcda1(0x258)];},Game_System[_0x22ff2b(0x3d4)]['setHelpWindowWordWrap']=function(_0xfe4962){const _0x4348f5=_0x22ff2b;if(this[_0x4348f5(0x1ad)]===undefined)this[_0x4348f5(0x35d)]();if(this['_MessageCoreSettings'][_0x4348f5(0x258)]===undefined)this[_0x4348f5(0x35d)]();this[_0x4348f5(0x1ad)][_0x4348f5(0x258)]=_0xfe4962;},Game_System['prototype'][_0x22ff2b(0x2cf)]=function(){const _0x2dc772=_0x22ff2b;if(this['_MessageCoreSettings']===undefined)this[_0x2dc772(0x35d)]();if(this['_MessageCoreSettings']['choiceLineHeight']===undefined)this[_0x2dc772(0x35d)]();return this[_0x2dc772(0x1ad)]['choiceLineHeight'];},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x380)]=function(_0x4254c1){const _0x1b4359=_0x22ff2b;if(this[_0x1b4359(0x1ad)]===undefined)this['initMessageCore']();if(this['_MessageCoreSettings'][_0x1b4359(0x3e8)]===undefined)this['initMessageCore']();this[_0x1b4359(0x1ad)][_0x1b4359(0x3e8)]=_0x4254c1||0x1;},Game_System['prototype'][_0x22ff2b(0x1de)]=function(){const _0x4621c4=_0x22ff2b;if(this[_0x4621c4(0x1ad)]===undefined)this[_0x4621c4(0x35d)]();if(this[_0x4621c4(0x1ad)][_0x4621c4(0x1da)]===undefined)this[_0x4621c4(0x35d)]();return this[_0x4621c4(0x1ad)][_0x4621c4(0x1da)];},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x3cf)]=function(_0x43a670){const _0x1888cb=_0x22ff2b;if(this[_0x1888cb(0x1ad)]===undefined)this[_0x1888cb(0x35d)]();if(this[_0x1888cb(0x1ad)][_0x1888cb(0x1da)]===undefined)this[_0x1888cb(0x35d)]();this[_0x1888cb(0x1ad)]['choiceRows']=_0x43a670||0x1;},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x2a6)]=function(){const _0x57acfa=_0x22ff2b;if(this[_0x57acfa(0x1ad)]===undefined)this[_0x57acfa(0x35d)]();if(this[_0x57acfa(0x1ad)][_0x57acfa(0x395)]===undefined)this['initMessageCore']();return this[_0x57acfa(0x1ad)][_0x57acfa(0x395)];},Game_System['prototype']['setChoiceListMaxColumns']=function(_0xc2e282){const _0x3ec7cd=_0x22ff2b;if(this[_0x3ec7cd(0x1ad)]===undefined)this[_0x3ec7cd(0x35d)]();if(this[_0x3ec7cd(0x1ad)][_0x3ec7cd(0x395)]===undefined)this[_0x3ec7cd(0x35d)]();this['_MessageCoreSettings']['choiceCols']=_0xc2e282||0x1;},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x1ae)]=function(){const _0x1e1fe6=_0x22ff2b;if(this[_0x1e1fe6(0x1ad)]===undefined)this[_0x1e1fe6(0x35d)]();if(this[_0x1e1fe6(0x1ad)][_0x1e1fe6(0x2f3)]===undefined)this[_0x1e1fe6(0x35d)]();return this[_0x1e1fe6(0x1ad)][_0x1e1fe6(0x2f3)];},Game_System[_0x22ff2b(0x3d4)][_0x22ff2b(0x248)]=function(_0x38dbdb){const _0x5b4cf6=_0x22ff2b;if(this['_MessageCoreSettings']===undefined)this[_0x5b4cf6(0x35d)]();if(this[_0x5b4cf6(0x1ad)]['choiceTextAlign']===undefined)this[_0x5b4cf6(0x35d)]();this[_0x5b4cf6(0x1ad)][_0x5b4cf6(0x2f3)]=_0x38dbdb[_0x5b4cf6(0x1af)]();},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x215)]=Game_Party[_0x22ff2b(0x3d4)][_0x22ff2b(0x2c1)],Game_Party[_0x22ff2b(0x3d4)][_0x22ff2b(0x2c1)]=function(){const _0x5a0fde=_0x22ff2b;VisuMZ['MessageCore'][_0x5a0fde(0x215)][_0x5a0fde(0x282)](this),this[_0x5a0fde(0x35d)]();},Game_Party[_0x22ff2b(0x3d4)]['initMessageCore']=function(){const _0x1b483f=_0x22ff2b;this[_0x1b483f(0x337)]={'type':0x0,'id':0x0,'quantity':0x0};},Game_Party[_0x22ff2b(0x3d4)][_0x22ff2b(0x3a4)]=function(){const _0x2b522f=_0x22ff2b;if(this[_0x2b522f(0x337)]===undefined)this[_0x2b522f(0x35d)]();return this[_0x2b522f(0x337)];},Game_Party[_0x22ff2b(0x3d4)]['setLastGainedItemData']=function(_0x5dd0ad,_0x2c6642){const _0x12b834=_0x22ff2b;if(this[_0x12b834(0x337)]===undefined)this[_0x12b834(0x35d)]();if(!_0x5dd0ad)return;if(DataManager[_0x12b834(0x365)](_0x5dd0ad)){if(_0x12b834(0x1bb)==='XozoH')this[_0x12b834(0x337)][_0x12b834(0x3ef)]=0x0;else return _0x3056f7=_0xec559a[_0x12b834(0x1d4)](/<(?:SHOW|HIDE|DISABLE|ENABLE)>/i,''),_0x4932ad=_0x13cde8['replace'](/<(?:SHOW|HIDE|DISABLE|ENABLE)[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i,''),_0x5e63b8=_0x3bdd1e[_0x12b834(0x1d4)](/<(?:SHOW|HIDE|DISABLE|ENABLE)[ ](?:ALL|ANY)[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i,''),_0x5285bf;}else{if(DataManager['isWeapon'](_0x5dd0ad)){if(_0x12b834(0x1c0)==='IfwPw')this['_lastGainedItemData']['type']=0x1;else{if(!_0x43809d[_0x12b834(0x35a)]())return'';let _0x1ff88e=_0x4c162a[_0x12b834(0x1b2)]||null;!_0x1ff88e&&_0x62855f[_0x12b834(0x408)]()&&(_0x1ff88e=_0x49f19e[_0x12b834(0x389)]());if(_0x1ff88e){let _0x4f0c20='';if(_0x1e0781)_0x4f0c20+=_0x12b834(0x24a)[_0x12b834(0x197)](_0x1ff88e['iconIndex']);return _0x4f0c20+=_0x1ff88e[_0x12b834(0x36d)],_0x4f0c20;}return'';}}else DataManager[_0x12b834(0x205)](_0x5dd0ad)&&(this[_0x12b834(0x337)][_0x12b834(0x3ef)]=0x2);}this[_0x12b834(0x337)]['id']=_0x5dd0ad['id'],this['_lastGainedItemData'][_0x12b834(0x1c4)]=_0x2c6642;},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x358)]=Game_Party[_0x22ff2b(0x3d4)][_0x22ff2b(0x280)],Game_Party[_0x22ff2b(0x3d4)][_0x22ff2b(0x280)]=function(_0xd10ce6,_0x5650a9,_0x46463a){const _0x22f454=_0x22ff2b;VisuMZ[_0x22f454(0x3d5)]['Game_Party_gainItem'][_0x22f454(0x282)](this,_0xd10ce6,_0x5650a9,_0x46463a),_0x5650a9>0x0&&this[_0x22f454(0x350)](_0xd10ce6,_0x5650a9);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x237)]=Game_Map[_0x22ff2b(0x3d4)][_0x22ff2b(0x2c1)],Game_Map[_0x22ff2b(0x3d4)]['initialize']=function(){const _0x4deedc=_0x22ff2b;VisuMZ['MessageCore']['Game_Map_initialize'][_0x4deedc(0x282)](this),this[_0x4deedc(0x2c2)]=[];},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x238)]=Game_Map['prototype']['setupEvents'],Game_Map[_0x22ff2b(0x3d4)][_0x22ff2b(0x2e8)]=function(){const _0xae4088=_0x22ff2b;VisuMZ['MessageCore']['Game_Map_setupEvents'][_0xae4088(0x282)](this),this[_0xae4088(0x2c2)]=[];},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x25f)]=Game_Map[_0x22ff2b(0x3d4)][_0x22ff2b(0x1a3)],Game_Map['prototype'][_0x22ff2b(0x1a3)]=function(){const _0x19d9ef=_0x22ff2b;VisuMZ['MessageCore'][_0x19d9ef(0x25f)][_0x19d9ef(0x282)](this),this['updateMessageCommonEvents']();},Game_Map[_0x22ff2b(0x3d4)][_0x22ff2b(0x1ef)]=function(_0x544659){const _0x2bb32b=_0x22ff2b;if(!$dataCommonEvents[_0x544659])return;this[_0x2bb32b(0x2c2)]=this[_0x2bb32b(0x2c2)]||[];const _0x421753=this[_0x2bb32b(0x226)]['_eventId'],_0x450cb7=new Game_MessageCommonEvent(_0x544659,_0x421753);this[_0x2bb32b(0x2c2)][_0x2bb32b(0x3ec)](_0x450cb7);},Game_Map[_0x22ff2b(0x3d4)][_0x22ff2b(0x25a)]=function(){const _0x59b8d6=_0x22ff2b;this[_0x59b8d6(0x2c2)]=this[_0x59b8d6(0x2c2)]||[];for(const _0x68a85f of this['_messageCommonEvents']){_0x59b8d6(0x3fc)===_0x59b8d6(0x3fc)?!_0x68a85f[_0x59b8d6(0x226)]?this['_messageCommonEvents'][_0x59b8d6(0x28a)](_0x68a85f):_0x68a85f[_0x59b8d6(0x26e)]():this['setWordWrap'](_0x28e20b[_0x59b8d6(0x272)]());}},Game_Interpreter[_0x22ff2b(0x3d4)][_0x22ff2b(0x1b6)]=function(_0x2d3af6){const _0x40fbb2=_0x22ff2b;if($gameMessage['isBusy']())return![];return this[_0x40fbb2(0x39f)](_0x2d3af6),this[_0x40fbb2(0x33f)](_0x2d3af6),this[_0x40fbb2(0x388)](_0x2d3af6),this[_0x40fbb2(0x2bc)](_0x40fbb2(0x328)),!![];},Game_Interpreter[_0x22ff2b(0x3d4)][_0x22ff2b(0x39f)]=function(_0x31bc4b){const _0x2aad59=_0x22ff2b;$gameMessage[_0x2aad59(0x245)](_0x31bc4b[0x0],_0x31bc4b[0x1]),$gameMessage['setBackground'](_0x31bc4b[0x2]),$gameMessage[_0x2aad59(0x1e3)](_0x31bc4b[0x3]),$gameMessage['setSpeakerName'](_0x31bc4b[0x4]);},Game_Interpreter[_0x22ff2b(0x3d4)]['addContinuousShowTextCommands']=function(_0x578b9f){const _0x23130b=_0x22ff2b;while(this[_0x23130b(0x307)]()){this[_0x23130b(0x3e2)]++;if(this[_0x23130b(0x2fb)]()[_0x23130b(0x3a9)]===0x191){if(_0x23130b(0x3e5)==='xQvRx')$gameMessage[_0x23130b(0x230)](this['currentCommand']()['parameters'][0x0]);else{const _0x216543=-(_0x33f5c3['floor'](_0x15f40d[_0x23130b(0x244)]-_0x3c5385[_0x23130b(0x26f)])/0x2),_0x35fbcc=_0x216543+_0x10fb5a[_0x23130b(0x244)]-this[_0x23130b(0x244)],_0x5648f0=-(_0xba7e51[_0x23130b(0x2db)](_0x3f4abb[_0x23130b(0x293)]-_0x1586f4[_0x23130b(0x36f)])/0x2),_0x3e139d=_0x5648f0+_0x49b8ba[_0x23130b(0x293)]-this['height'];this['x']=this['x'][_0x23130b(0x3a7)](_0x216543,_0x35fbcc),this['y']=this['y']['clamp'](_0x5648f0,_0x3e139d);}}if(this[_0x23130b(0x27f)]())break;}},Game_Interpreter[_0x22ff2b(0x3d4)][_0x22ff2b(0x307)]=function(){const _0x2a23ca=_0x22ff2b;if(this[_0x2a23ca(0x2c4)]()===0x65&&$gameSystem[_0x2a23ca(0x3ce)]()>0x4){if(_0x2a23ca(0x38a)===_0x2a23ca(0x38a))return!![];else _0x33a864['MessageCore']['Window_Base_processEscapeCharacter'][_0x2a23ca(0x282)](this,_0x2dfb9d,_0x260039);}else return this['nextEventCode']()===0x191;},Game_Interpreter[_0x22ff2b(0x3d4)][_0x22ff2b(0x27f)]=function(){const _0x1f72ef=_0x22ff2b;return $gameMessage[_0x1f72ef(0x336)][_0x1f72ef(0x2fe)]>=$gameSystem[_0x1f72ef(0x3ce)]()&&this[_0x1f72ef(0x2c4)]()!==0x191;},Game_Interpreter['prototype'][_0x22ff2b(0x388)]=function(_0x2c1044){const _0x31c0de=_0x22ff2b;switch(this[_0x31c0de(0x2c4)]()){case 0x66:this[_0x31c0de(0x3e2)]++,this[_0x31c0de(0x400)](this[_0x31c0de(0x2fb)]()[_0x31c0de(0x22c)]);break;case 0x67:this[_0x31c0de(0x3e2)]++,this[_0x31c0de(0x2d8)](this[_0x31c0de(0x2fb)]()['parameters']);break;case 0x68:this[_0x31c0de(0x3e2)]++,this[_0x31c0de(0x1ee)](this[_0x31c0de(0x2fb)]()[_0x31c0de(0x22c)]);break;}},VisuMZ[_0x22ff2b(0x3d5)]['Game_Interpreter_setupChoices']=Game_Interpreter[_0x22ff2b(0x3d4)]['setupChoices'],Game_Interpreter['prototype'][_0x22ff2b(0x400)]=function(_0x3f7f5d){const _0x41f21a=_0x22ff2b;_0x3f7f5d=this[_0x41f21a(0x35f)](),VisuMZ[_0x41f21a(0x3d5)][_0x41f21a(0x259)][_0x41f21a(0x282)](this,_0x3f7f5d);},Game_Interpreter['prototype'][_0x22ff2b(0x35f)]=function(){const _0x57aa7b=_0x22ff2b,_0x5aa2f8=this[_0x57aa7b(0x3e2)],_0x1b588b=[];let _0x40942f=0x0;this[_0x57aa7b(0x3e2)]++;while(this[_0x57aa7b(0x3e2)]<this[_0x57aa7b(0x347)][_0x57aa7b(0x2fe)]){if(this['currentCommand']()[_0x57aa7b(0x23b)]===this[_0x57aa7b(0x1db)]){if(this[_0x57aa7b(0x2fb)]()[_0x57aa7b(0x3a9)]===0x194&&this[_0x57aa7b(0x2c4)]()!==0x66)break;else{if(this[_0x57aa7b(0x2fb)]()['code']===0x66)this['adjustShowChoiceExtension'](_0x40942f,this[_0x57aa7b(0x2fb)](),_0x5aa2f8),this['_index']-=0x2;else this['currentCommand']()[_0x57aa7b(0x3a9)]===0x192&&(this['currentCommand']()[_0x57aa7b(0x22c)][0x0]=_0x40942f,_0x40942f++);}}this['_index']++;}return this[_0x57aa7b(0x3e2)]=_0x5aa2f8,this[_0x57aa7b(0x2fb)]()[_0x57aa7b(0x22c)];},Game_Interpreter[_0x22ff2b(0x3d4)][_0x22ff2b(0x2f2)]=function(_0x2ceda2,_0x11a3e5,_0x2b655d){const _0x34b366=_0x22ff2b;this[_0x34b366(0x214)](_0x2ceda2,_0x11a3e5,_0x2b655d),this[_0x34b366(0x32b)](_0x2ceda2,_0x11a3e5,_0x2b655d),this[_0x34b366(0x3c0)](_0x11a3e5,_0x2b655d);},Game_Interpreter[_0x22ff2b(0x3d4)]['adjustShowChoiceDefault']=function(_0x28d8e2,_0x4f6167,_0x4ef2fe){const _0xbe248a=_0x22ff2b;if(_0x4f6167[_0xbe248a(0x22c)][0x2]<0x0)return;const _0x5e0553=_0x4f6167[_0xbe248a(0x22c)][0x2]+_0x28d8e2;this[_0xbe248a(0x347)][_0x4ef2fe]['parameters'][0x2]=_0x5e0553;},Game_Interpreter['prototype'][_0x22ff2b(0x32b)]=function(_0x2e48e5,_0xe2441e,_0x538c9a){const _0x45650d=_0x22ff2b;if(_0xe2441e['parameters'][0x1]>=0x0){if(_0x45650d(0x373)!=='YEllM'){var _0x27f67e=_0xe2441e[_0x45650d(0x22c)][0x1]+_0x2e48e5;this[_0x45650d(0x347)][_0x538c9a][_0x45650d(0x22c)][0x1]=_0x27f67e;}else this['textSpeed']=_0x4044ce[_0x45650d(0x3d5)][_0x45650d(0x236)][_0x45650d(0x219)][_0x45650d(0x3cc)];}else _0xe2441e[_0x45650d(0x22c)][0x1]===-0x2&&('ZiHGj'===_0x45650d(0x249)?this[_0x45650d(0x347)][_0x538c9a]['parameters'][0x1]=_0xe2441e[_0x45650d(0x22c)][0x1]:_0x801830=_0x3f55e1[_0x45650d(0x389)]());},Game_Interpreter['prototype'][_0x22ff2b(0x3c0)]=function(_0x4790fb,_0x36bc97){const _0x3fd4f9=_0x22ff2b;for(const _0x3e972c of _0x4790fb['parameters'][0x0]){_0x3fd4f9(0x1eb)!==_0x3fd4f9(0x288)?this[_0x3fd4f9(0x347)][_0x36bc97][_0x3fd4f9(0x22c)][0x0][_0x3fd4f9(0x3ec)](_0x3e972c):_0x20081f[_0x3fd4f9(0x323)]=new _0x3514de(_0x3fd4f9(0x326)+_0x23d5bf[_0x3fd4f9(0x204)][_0x3fd4f9(0x1d4)](/\\/g,'\x1b')+'\x27');}this[_0x3fd4f9(0x347)][_0x3fd4f9(0x3bb)](this[_0x3fd4f9(0x3e2)]-0x1,0x2);};function Game_MessageCommonEvent(){const _0x2a28b9=_0x22ff2b;this[_0x2a28b9(0x2c1)](...arguments);}function _0x1467(_0x3fba1b,_0x14d6f1){const _0x136329=_0x1363();return _0x1467=function(_0x146779,_0x4a8fa5){_0x146779=_0x146779-0x194;let _0x5164ed=_0x136329[_0x146779];return _0x5164ed;},_0x1467(_0x3fba1b,_0x14d6f1);}Game_MessageCommonEvent['prototype'][_0x22ff2b(0x2c1)]=function(_0x2c0c9d,_0x4aa8de){const _0x3deeaf=_0x22ff2b;this[_0x3deeaf(0x24f)]=_0x2c0c9d,this[_0x3deeaf(0x352)]=_0x4aa8de||0x0,this[_0x3deeaf(0x2b1)]();},Game_MessageCommonEvent[_0x22ff2b(0x3d4)][_0x22ff2b(0x3e9)]=function(){return $dataCommonEvents[this['_commonEventId']];},Game_MessageCommonEvent[_0x22ff2b(0x3d4)][_0x22ff2b(0x209)]=function(){const _0x512e2d=_0x22ff2b;return this[_0x512e2d(0x3e9)]()[_0x512e2d(0x209)];},Game_MessageCommonEvent['prototype']['refresh']=function(){const _0x26d58c=_0x22ff2b;this['_interpreter']=new Game_Interpreter(),this['_interpreter'][_0x26d58c(0x281)](this[_0x26d58c(0x209)](),this['_eventId']);},Game_MessageCommonEvent[_0x22ff2b(0x3d4)][_0x22ff2b(0x26e)]=function(){const _0x4b2606=_0x22ff2b;this[_0x4b2606(0x226)]&&('UTgIG'!=='jOWEI'?this[_0x4b2606(0x226)]['isRunning']()?'dPwZg'!==_0x4b2606(0x1ed)?(this[_0x4b2606(0x1d3)][_0x4b2606(0x3ff)]-=_0xf570d0[_0x4b2606(0x3d5)]['Settings'][_0x4b2606(0x2da)][_0x4b2606(0x41b)],this['contents']['fontSize']=_0x5112c6[_0x4b2606(0x2d3)](this[_0x4b2606(0x1d3)][_0x4b2606(0x3ff)],_0x2f6963['MessageCore']['Settings'][_0x4b2606(0x2da)]['FontSmallerCap'])):this[_0x4b2606(0x226)][_0x4b2606(0x26e)]():this[_0x4b2606(0x3d7)]():this['initialize'](...arguments));},Game_MessageCommonEvent['prototype']['clear']=function(){const _0x46fc95=_0x22ff2b;this[_0x46fc95(0x226)]=null;},Scene_Message[_0x22ff2b(0x3d4)]['messageWindowRect']=function(){const _0x49ea70=_0x22ff2b,_0x2ecae8=Math[_0x49ea70(0x2cb)](Graphics['width'],$gameSystem[_0x49ea70(0x3c1)]()),_0xd2410b=$gameSystem[_0x49ea70(0x3ce)](),_0x1e4837=this[_0x49ea70(0x2a3)](_0xd2410b,![]),_0x3e8af7=(Graphics[_0x49ea70(0x26f)]-_0x2ecae8)/0x2,_0x95a45a=0x0;return new Rectangle(_0x3e8af7,_0x95a45a,_0x2ecae8,_0x1e4837);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x346)]=Scene_Options[_0x22ff2b(0x3d4)][_0x22ff2b(0x22f)],Scene_Options[_0x22ff2b(0x3d4)]['maxCommands']=function(){const _0x2ae532=_0x22ff2b;let _0xb5f0f3=VisuMZ[_0x2ae532(0x3d5)][_0x2ae532(0x346)]['call'](this);const _0x192f7e=VisuMZ[_0x2ae532(0x3d5)]['Settings'];if(_0x192f7e['TextSpeed']['AddOption']&&_0x192f7e[_0x2ae532(0x219)]['AdjustRect'])_0xb5f0f3++;return _0xb5f0f3;},VisuMZ['MessageCore'][_0x22ff2b(0x3f5)]=Window_Base[_0x22ff2b(0x3d4)]['initialize'],Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x2c1)]=function(_0x47f180){const _0x405482=_0x22ff2b;this[_0x405482(0x35d)](_0x47f180),VisuMZ['MessageCore'][_0x405482(0x3f5)][_0x405482(0x282)](this,_0x47f180);},Window_Base[_0x22ff2b(0x3d4)]['initMessageCore']=function(_0x5dc40c){const _0x3f63da=_0x22ff2b;this['initTextAlignement'](),this['resetWordWrap'](),this[_0x3f63da(0x34a)](_0x5dc40c);},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x3eb)]=function(){const _0x3de5a1=_0x22ff2b;this[_0x3de5a1(0x31f)]('default');},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x31f)]=function(_0x462d0d){this['_textAlignment']=_0x462d0d;},Window_Base['prototype']['getTextAlignment']=function(){const _0x187188=_0x22ff2b;return this[_0x187188(0x37e)];},VisuMZ[_0x22ff2b(0x3d5)]['Window_Base_textSizeEx']=Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x37d)],Window_Base[_0x22ff2b(0x3d4)]['textSizeEx']=function(_0x55c42){const _0x51917d=_0x22ff2b;return this[_0x51917d(0x33b)](),VisuMZ[_0x51917d(0x3d5)]['Window_Base_textSizeEx'][_0x51917d(0x282)](this,_0x55c42);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x303)]=Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x3b1)],Window_Base[_0x22ff2b(0x3d4)]['processAllText']=function(_0x2ec84e){const _0x4c8162=_0x22ff2b;VisuMZ[_0x4c8162(0x3d5)][_0x4c8162(0x303)][_0x4c8162(0x282)](this,_0x2ec84e);if(_0x2ec84e[_0x4c8162(0x3f6)])this['setTextAlignment'](_0x4c8162(0x1d1));},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x33b)]=function(){const _0x5eec1e=_0x22ff2b;this[_0x5eec1e(0x3af)](![]);},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x375)]=function(){const _0x463060=_0x22ff2b;return this[_0x463060(0x23c)];},Window_Base[_0x22ff2b(0x3d4)]['setWordWrap']=function(_0x4703e0){const _0x2a2f0a=_0x22ff2b;return this[_0x2a2f0a(0x23c)]=_0x4703e0,'';},Window_Base['prototype']['registerResetRect']=function(_0x4f09b3){const _0x4b9bcf=_0x22ff2b;this[_0x4b9bcf(0x29f)]=JsonEx[_0x4b9bcf(0x224)](_0x4f09b3);},Window_Base['prototype'][_0x22ff2b(0x1df)]=function(){const _0x3e14e0=_0x22ff2b;this[_0x3e14e0(0x1d3)][_0x3e14e0(0x195)]=$gameSystem[_0x3e14e0(0x38d)](),this[_0x3e14e0(0x1d3)][_0x3e14e0(0x3ff)]=$gameSystem[_0x3e14e0(0x2f8)](),this[_0x3e14e0(0x1d3)][_0x3e14e0(0x316)]=![],this[_0x3e14e0(0x1d3)][_0x3e14e0(0x243)]=![],this[_0x3e14e0(0x3bc)]();},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x3bc)]=function(){const _0x1f4992=_0x22ff2b;this[_0x1f4992(0x300)](ColorManager[_0x1f4992(0x309)]()),this[_0x1f4992(0x2e9)](ColorManager[_0x1f4992(0x19d)]());const _0x2e1d00=VisuMZ[_0x1f4992(0x3d5)]['Settings'][_0x1f4992(0x2da)];_0x2e1d00['DefaultOutlineWidth']===undefined&&(_0x1f4992(0x413)===_0x1f4992(0x413)?_0x2e1d00['DefaultOutlineWidth']=0x3:(this[_0x1f4992(0x244)]=_0x47adf9[_0x1f4992(0x2cb)](this[_0x1f4992(0x244)],_0x26ab00['width']),this[_0x1f4992(0x293)]=_0x5de5a9[_0x1f4992(0x2cb)](this[_0x1f4992(0x293)],_0xf991ec[_0x1f4992(0x293)]))),this[_0x1f4992(0x1d3)][_0x1f4992(0x221)]=_0x2e1d00[_0x1f4992(0x26d)],this[_0x1f4992(0x32d)](![]);},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x32d)]=function(_0x26a548){const _0x2a8ee9=_0x22ff2b;this[_0x2a8ee9(0x2f0)]=_0x26a548;},Window_Base[_0x22ff2b(0x3d4)]['isColorLocked']=function(){return this['_colorLock'];},Window_Base['prototype'][_0x22ff2b(0x223)]=function(){return![];},Window_Base['prototype'][_0x22ff2b(0x2c0)]=function(){const _0x523eb7=_0x22ff2b,_0x360475=[_0x523eb7(0x195),_0x523eb7(0x3ff),_0x523eb7(0x316),_0x523eb7(0x243),_0x523eb7(0x3db),_0x523eb7(0x36c),_0x523eb7(0x221),_0x523eb7(0x374)];let _0x4cbe57={};for(const _0x717a9b of _0x360475){_0x4cbe57[_0x717a9b]=this[_0x523eb7(0x1d3)][_0x717a9b];}return _0x4cbe57;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x28f)]=function(_0x295c00){const _0x50b06e=_0x22ff2b;for(const _0x369a5b in _0x295c00){if(_0x50b06e(0x368)!=='qHBDx')return!![];else this[_0x50b06e(0x1d3)][_0x369a5b]=_0x295c00[_0x369a5b];}},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x252)]=Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x26e)],Window_Base['prototype'][_0x22ff2b(0x26e)]=function(){const _0x114378=_0x22ff2b;VisuMZ['MessageCore'][_0x114378(0x252)][_0x114378(0x282)](this),this['updateMove']();},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x2dc)]=function(){return![];},Window_Base['prototype'][_0x22ff2b(0x344)]=function(){const _0x4dd7c2=_0x22ff2b;this['_moveDuration']>0x0&&(this[_0x4dd7c2(0x2dc)]()&&(this['x']=this['applyMoveEasing'](this['x'],this[_0x4dd7c2(0x2a0)]),this['y']=this['applyMoveEasing'](this['y'],this[_0x4dd7c2(0x403)]),this[_0x4dd7c2(0x244)]=this['applyMoveEasing'](this[_0x4dd7c2(0x244)],this['_moveTargetWidth']),this[_0x4dd7c2(0x293)]=this[_0x4dd7c2(0x284)](this[_0x4dd7c2(0x293)],this[_0x4dd7c2(0x20f)]),this[_0x4dd7c2(0x1f4)]()),this[_0x4dd7c2(0x20e)]--);},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x1f4)]=function(_0x4b7f40,_0x1e4fb7){const _0x5bfe13=_0x22ff2b;if(!_0x4b7f40){if('VcDlE'!==_0x5bfe13(0x3cb)){this[_0x5bfe13(0x2a0)]=_0xb811d7,this[_0x5bfe13(0x403)]=_0x295c9f,this[_0x5bfe13(0x32a)]=_0x2926f1||this[_0x5bfe13(0x244)],this['_moveTargetHeight']=_0x597271||this[_0x5bfe13(0x293)],this['_moveDuration']=_0x195c16||0x1;if(this[_0x5bfe13(0x20e)]<=0x0)this['_moveDuration']=0x1;this[_0x5bfe13(0x220)]=this[_0x5bfe13(0x20e)],this[_0x5bfe13(0x22a)]=_0x48064f||0x0;if(_0xa811d7<=0x0)this['updateMove']();}else this[_0x5bfe13(0x244)]=Math['min'](this[_0x5bfe13(0x244)],Graphics[_0x5bfe13(0x244)]),this[_0x5bfe13(0x293)]=Math[_0x5bfe13(0x2cb)](this[_0x5bfe13(0x293)],Graphics['height']);}if(!_0x1e4fb7){const _0x1228d6=-(Math[_0x5bfe13(0x2db)](Graphics[_0x5bfe13(0x244)]-Graphics[_0x5bfe13(0x26f)])/0x2),_0x507853=_0x1228d6+Graphics[_0x5bfe13(0x244)]-this[_0x5bfe13(0x244)],_0x160f7b=-(Math[_0x5bfe13(0x2db)](Graphics[_0x5bfe13(0x293)]-Graphics['boxHeight'])/0x2),_0x3af9f=_0x160f7b+Graphics['height']-this['height'];this['x']=this['x'][_0x5bfe13(0x3a7)](_0x1228d6,_0x507853),this['y']=this['y'][_0x5bfe13(0x3a7)](_0x160f7b,_0x3af9f);}},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x284)]=function(_0x33e2df,_0x4a3ddf){const _0xbd19cb=_0x22ff2b,_0x432905=this['_moveDuration'],_0x16b7ac=this[_0xbd19cb(0x220)],_0x331851=this['calcMoveEasing']((_0x16b7ac-_0x432905)/_0x16b7ac),_0x13a4d4=this[_0xbd19cb(0x1f5)]((_0x16b7ac-_0x432905+0x1)/_0x16b7ac),_0xa11eba=(_0x33e2df-_0x4a3ddf*_0x331851)/(0x1-_0x331851);return _0xa11eba+(_0x4a3ddf-_0xa11eba)*_0x13a4d4;},Window_Base['prototype'][_0x22ff2b(0x1f5)]=function(_0x488d25){const _0x1b1715=_0x22ff2b,_0x593f66=0x2;switch(this[_0x1b1715(0x22a)]){case 0x0:return _0x488d25;case 0x1:return this[_0x1b1715(0x3a3)](_0x488d25,_0x593f66);case 0x2:return this[_0x1b1715(0x2f7)](_0x488d25,_0x593f66);case 0x3:return this[_0x1b1715(0x3ed)](_0x488d25,_0x593f66);default:return Imported[_0x1b1715(0x1dc)]?VisuMZ[_0x1b1715(0x284)](_0x488d25,this[_0x1b1715(0x22a)]):_0x488d25;}},Window_Base['prototype'][_0x22ff2b(0x2d5)]=function(_0x5bfb5b,_0x438d9a,_0x513587,_0x520169,_0x4b7ad5,_0x57a1fc){const _0x558db1=_0x22ff2b;this[_0x558db1(0x2a0)]=_0x5bfb5b,this[_0x558db1(0x403)]=_0x438d9a,this['_moveTargetWidth']=_0x513587||this[_0x558db1(0x244)],this['_moveTargetHeight']=_0x520169||this['height'],this[_0x558db1(0x20e)]=_0x4b7ad5||0x1;if(this[_0x558db1(0x20e)]<=0x0)this[_0x558db1(0x20e)]=0x1;this[_0x558db1(0x220)]=this['_moveDuration'],this[_0x558db1(0x22a)]=_0x57a1fc||0x0;if(_0x4b7ad5<=0x0)this[_0x558db1(0x344)]();},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x33e)]=function(_0xdea8a5,_0x1a30db,_0x3d2674,_0x3e4f55,_0x5f3c38,_0x37684a){const _0x3d85e6=_0x22ff2b;this[_0x3d85e6(0x2a0)]=this['x']+_0xdea8a5,this['_moveTargetY']=this['y']+_0x1a30db,this['_moveTargetWidth']=this[_0x3d85e6(0x244)]+(_0x3d2674||0x0),this[_0x3d85e6(0x20f)]=this['height']+(_0x3e4f55||0x0),this[_0x3d85e6(0x20e)]=_0x5f3c38||0x1;if(this[_0x3d85e6(0x20e)]<=0x0)this[_0x3d85e6(0x20e)]=0x1;this[_0x3d85e6(0x220)]=this[_0x3d85e6(0x20e)],this[_0x3d85e6(0x22a)]=_0x37684a||0x0;if(_0x5f3c38<=0x0)this[_0x3d85e6(0x344)]();},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x394)]=function(_0xed716d,_0x4a5932){const _0x5244df=_0x22ff2b;this['moveTo'](this[_0x5244df(0x29f)]['x'],this['_resetRect']['y'],this[_0x5244df(0x29f)][_0x5244df(0x244)],this['_resetRect']['height'],_0xed716d,_0x4a5932);},VisuMZ['MessageCore'][_0x22ff2b(0x3c5)]=Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x300)],Window_Base[_0x22ff2b(0x3d4)]['changeTextColor']=function(_0x5e470b){const _0x2114bc=_0x22ff2b;if(this[_0x2114bc(0x247)]())return;_0x5e470b=_0x5e470b['replace'](/\,/g,''),this[_0x2114bc(0x2d2)]=this['_textColorStack']||[],this[_0x2114bc(0x2d2)][_0x2114bc(0x34e)](this[_0x2114bc(0x1d3)][_0x2114bc(0x3db)]),VisuMZ[_0x2114bc(0x3d5)]['Window_Base_changeTextColor'][_0x2114bc(0x282)](this,_0x5e470b);},Window_Base['prototype'][_0x22ff2b(0x2bb)]=function(_0x32db9b){const _0xa3777c=_0x22ff2b;this[_0xa3777c(0x2ab)](_0x32db9b);if(this[_0xa3777c(0x247)]())return;_0x32db9b[_0xa3777c(0x3f6)]&&(this['_textColorStack']=this['_textColorStack']||[],this[_0xa3777c(0x1d3)][_0xa3777c(0x3db)]=this['_textColorStack']['shift']()||ColorManager[_0xa3777c(0x309)]());},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x1f9)]=function(_0x263ac5){const _0x46f357=_0x22ff2b;return _0x263ac5=this[_0x46f357(0x377)](_0x263ac5),_0x263ac5=this[_0x46f357(0x2f5)](_0x263ac5),_0x263ac5=this[_0x46f357(0x3f0)](_0x263ac5),_0x263ac5=this[_0x46f357(0x304)](_0x263ac5),_0x263ac5=this[_0x46f357(0x1a0)](_0x263ac5),_0x263ac5=this[_0x46f357(0x315)](_0x263ac5),_0x263ac5=this[_0x46f357(0x30f)](_0x263ac5),_0x263ac5=this[_0x46f357(0x361)](_0x263ac5),_0x263ac5=this[_0x46f357(0x386)](_0x263ac5),_0x263ac5=this[_0x46f357(0x200)](_0x263ac5),_0x263ac5=this[_0x46f357(0x360)](_0x263ac5),_0x263ac5=this['convertMessageCoreEscapeReplacements'](_0x263ac5),_0x263ac5=this['postConvertEscapeCharacters'](_0x263ac5),_0x263ac5=this[_0x46f357(0x3f0)](_0x263ac5),_0x263ac5=this[_0x46f357(0x3b9)](_0x263ac5),_0x263ac5=this[_0x46f357(0x1c2)](_0x263ac5),_0x263ac5;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x377)]=function(_0x2e7d1a){const _0x15dda5=_0x22ff2b;for(const _0x3949dc of VisuMZ[_0x15dda5(0x3d5)][_0x15dda5(0x236)][_0x15dda5(0x295)]){if(_0x15dda5(0x206)!==_0x15dda5(0x34b))_0x2e7d1a[_0x15dda5(0x1f2)](_0x3949dc[_0x15dda5(0x2ea)])&&(_0x2e7d1a=_0x2e7d1a['replace'](_0x3949dc['textCodeCheck'],_0x3949dc['textCodeResult'][_0x15dda5(0x1ff)](this)));else{const _0x17b935=this[_0x15dda5(0x1f6)](_0x30c8ec),_0x1f7205=_0x5b40b2[_0x15dda5(0x1ae)]()!==_0x15dda5(0x1d1)?_0x15dda5(0x3f9)[_0x15dda5(0x197)](_0x4b0ec9[_0x15dda5(0x1ae)]()):'',_0x1246e5=_0x1f7205+this[_0x15dda5(0x24e)](_0x560172);this[_0x15dda5(0x3f4)](this[_0x15dda5(0x3aa)](_0x5c0fab));const _0x952d04=this['textSizeEx'](_0x1246e5)[_0x15dda5(0x293)],_0x58fdab=_0x1dabaf[_0x15dda5(0x2d3)](_0x17b935['y'],_0x17b935['y']+_0x37aed5['round']((_0x17b935[_0x15dda5(0x293)]-_0x952d04)/0x2));this[_0x15dda5(0x2f1)](_0x1246e5,_0x17b935['x'],_0x58fdab,_0x17b935[_0x15dda5(0x244)]);}}return _0x2e7d1a;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x2f5)]=function(_0x554888){const _0x502776=_0x22ff2b;return _0x554888=_0x554888[_0x502776(0x1d4)](/\\/g,'\x1b'),_0x554888=_0x554888[_0x502776(0x1d4)](/\x1b\x1b/g,'\x5c'),_0x554888;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x3f0)]=function(_0x524073){const _0x3a9d69=_0x22ff2b;for(;;){if(_0x524073[_0x3a9d69(0x1f2)](/\\V\[(\d+)\]/gi)){if(_0x3a9d69(0x330)==='zncPO')_0x524073=_0x524073[_0x3a9d69(0x1d4)](/\\V\[(\d+)\]/gi,(_0x32c0c5,_0x1933a8)=>this['convertBackslashCharacters'](String($gameVariables['value'](parseInt(_0x1933a8)))));else{const _0x18105c=_0x45466[_0x3a9d69(0x38f)]('['+_0x2fbd98['$1'][_0x3a9d69(0x1f2)](/\d+/g)+']');for(const _0x1150c0 of _0x18105c){if(!_0x1ff355['value'](_0x1150c0))return!![];}return![];}}else{if(_0x524073[_0x3a9d69(0x1f2)](/\x1bV\[(\d+)\]/gi)){if(_0x3a9d69(0x2cd)!==_0x3a9d69(0x2cd))return _0x24448f=_0x48a548['replace'](/\x1bN\[(\d+)\]/gi,(_0x1147cb,_0xf24685)=>this['actorName'](_0x4bf7d2(_0xf24685))),_0x3f9fd3=_0x240298[_0x3a9d69(0x1d4)](/\x1bP\[(\d+)\]/gi,(_0x4e13d3,_0x3220c6)=>this[_0x3a9d69(0x294)](_0x4a5ba9(_0x3220c6))),_0xfa3f5f=_0x2068ac['replace'](/\x1bG/gi,_0x2f0950['currencyUnit']),_0x10268d;else _0x524073=_0x524073['replace'](/\x1bV\[(\d+)\]/gi,(_0x362e20,_0x46d1ca)=>this[_0x3a9d69(0x2f5)](String($gameVariables[_0x3a9d69(0x290)](parseInt(_0x46d1ca)))));}else break;}}return _0x524073;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x304)]=function(_0x23cdda){const _0xdec1d6=_0x22ff2b;return this[_0xdec1d6(0x312)](),_0x23cdda;},Window_Base[_0x22ff2b(0x3d4)]['postConvertEscapeCharacters']=function(_0x3cb8f6){return _0x3cb8f6;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x1a0)]=function(_0x31372){const _0x22e230=_0x22ff2b;return _0x31372=_0x31372[_0x22e230(0x1d4)](/<(?:SHOW|HIDE|DISABLE|ENABLE)>/i,''),_0x31372=_0x31372['replace'](/<(?:SHOW|HIDE|DISABLE|ENABLE)[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i,''),_0x31372=_0x31372[_0x22e230(0x1d4)](/<(?:SHOW|HIDE|DISABLE|ENABLE)[ ](?:ALL|ANY)[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i,''),_0x31372;},Window_Base['prototype']['convertFontSettingsEscapeCharacters']=function(_0x2989eb){const _0x148e2b=_0x22ff2b;return _0x2989eb=_0x2989eb['replace'](/<B>/gi,'\x1bBOLD[1]'),_0x2989eb=_0x2989eb[_0x148e2b(0x1d4)](/<\/B>/gi,_0x148e2b(0x30b)),_0x2989eb=_0x2989eb[_0x148e2b(0x1d4)](/<I>/gi,_0x148e2b(0x2ff)),_0x2989eb=_0x2989eb[_0x148e2b(0x1d4)](/<\/I>/gi,_0x148e2b(0x1f3)),_0x2989eb;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x30f)]=function(_0x51c229){const _0x40eb04=_0x22ff2b;return _0x51c229=_0x51c229['replace'](/<LEFT>/gi,'\x1bTEXTALIGNMENT[1]'),_0x51c229=_0x51c229[_0x40eb04(0x1d4)](/<\/LEFT>/gi,'\x1bTEXTALIGNMENT[0]'),_0x51c229=_0x51c229[_0x40eb04(0x1d4)](/<CENTER>/gi,_0x40eb04(0x2a7)),_0x51c229=_0x51c229[_0x40eb04(0x1d4)](/<\/CENTER>/gi,_0x40eb04(0x339)),_0x51c229=_0x51c229[_0x40eb04(0x1d4)](/<RIGHT>/gi,_0x40eb04(0x1c1)),_0x51c229=_0x51c229['replace'](/<\/RIGHT>/gi,_0x40eb04(0x339)),_0x51c229;},Window_Base[_0x22ff2b(0x3d4)]['convertLockColorsEscapeCharacters']=function(_0x145acb){const _0x41bfe6=_0x22ff2b;return _0x145acb=_0x145acb[_0x41bfe6(0x1d4)](/<COLORLOCK>/gi,'\x1bCOLORLOCK[1]'),_0x145acb=_0x145acb['replace'](/<\/COLORLOCK>/gi,_0x41bfe6(0x3f7)),_0x145acb=_0x145acb['replace'](/\(\(\(/gi,_0x41bfe6(0x21c)),_0x145acb=_0x145acb[_0x41bfe6(0x1d4)](/\)\)\)/gi,_0x41bfe6(0x3f7)),_0x145acb;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x386)]=function(_0x57517b){const _0x54f947=_0x22ff2b;return _0x57517b=_0x57517b[_0x54f947(0x1d4)](/\x1bN\[(\d+)\]/gi,(_0x55ec47,_0x57a217)=>this['actorName'](parseInt(_0x57a217))),_0x57517b=_0x57517b[_0x54f947(0x1d4)](/\x1bP\[(\d+)\]/gi,(_0x26bfcb,_0x4ff419)=>this['partyMemberName'](parseInt(_0x4ff419))),_0x57517b=_0x57517b['replace'](/\x1bG/gi,TextManager[_0x54f947(0x2e3)]),_0x57517b;},Window_Base[_0x22ff2b(0x3d4)]['convertHardcodedEscapeReplacements']=function(_0x433439){const _0x561939=_0x22ff2b;return _0x433439=_0x433439[_0x561939(0x1d4)](/\<(?:BATTLE|CURRENT BATTLE) TARGET\>/gi,this['battleTargetName']()),_0x433439=_0x433439[_0x561939(0x1d4)](/\<(?:BATTLE|CURRENT BATTLE) (?:USER|SUBJECT)\>/gi,this['battleUserName']()),_0x433439=_0x433439[_0x561939(0x1d4)](/\<(?:BATTLE|CURRENT BATTLE) (?:ITEM|SKILL|ACTION)\>/gi,this[_0x561939(0x298)](!![])),_0x433439=_0x433439[_0x561939(0x1d4)](/\<(?:BATTLE|CURRENT BATTLE) (?:ITEM|SKILL|ACTION) NAME\>/gi,this[_0x561939(0x298)](![])),_0x433439;},Window_Base['prototype']['battleTargetName']=function(){const _0x16d206=_0x22ff2b;if(!SceneManager['isSceneBattle']())return'';if(BattleManager[_0x16d206(0x359)])return BattleManager[_0x16d206(0x359)][_0x16d206(0x36d)]();if(BattleManager[_0x16d206(0x392)][0x0])return BattleManager['_targets'][0x0]['name']();return'';},Window_Base['prototype'][_0x22ff2b(0x1a7)]=function(){const _0x56b1bc=_0x22ff2b;if(!SceneManager[_0x56b1bc(0x35a)]())return'';let _0x39e3d2=null;return _0x39e3d2=BattleManager['_subject'],!_0x39e3d2&&BattleManager[_0x56b1bc(0x408)]()&&(_0x39e3d2=BattleManager[_0x56b1bc(0x2c5)]()),_0x39e3d2?_0x39e3d2[_0x56b1bc(0x36d)]():'';},Window_Base['prototype'][_0x22ff2b(0x298)]=function(_0x103e50){const _0x42b297=_0x22ff2b;if(!SceneManager[_0x42b297(0x35a)]())return'';let _0x21477b=BattleManager[_0x42b297(0x1b2)]||null;if(!_0x21477b&&BattleManager['isInputting']()){if(_0x42b297(0x246)===_0x42b297(0x246))_0x21477b=BattleManager[_0x42b297(0x389)]();else{if(this[_0x42b297(0x1ad)]===_0x334010)this[_0x42b297(0x35d)]();if(this[_0x42b297(0x1ad)][_0x42b297(0x3e8)]===_0x3c0372)this[_0x42b297(0x35d)]();this[_0x42b297(0x1ad)][_0x42b297(0x3e8)]=_0x57bf70||0x1;}}if(_0x21477b){let _0x2cdcdc='';if(_0x103e50)_0x2cdcdc+=_0x42b297(0x24a)[_0x42b297(0x197)](_0x21477b[_0x42b297(0x2be)]);return _0x2cdcdc+=_0x21477b[_0x42b297(0x36d)],_0x2cdcdc;}return'';},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x360)]=function(_0x3e4bd7){const _0x42de5a=_0x22ff2b;for(const _0x5c9e19 of VisuMZ[_0x42de5a(0x3d5)][_0x42de5a(0x236)][_0x42de5a(0x2ec)]){if(_0x42de5a(0x2fa)==='xeEpg'){if(_0x109774<=0x0)return;const _0x4cf89c=_0x3f2944[_0x42de5a(0x3d5)][_0x42de5a(0x236)][_0x42de5a(0x410)][_0x42de5a(0x2ee)+_0x4ea95c];let _0x48cbb4=_0x37c4ad['name'][_0x42de5a(0x415)]();if(/^\d+$/['test'](_0x48cbb4))return;if(_0x52d489['MessageCore']['AutoColorBypassList'][_0x42de5a(0x262)](_0x48cbb4[_0x42de5a(0x2a4)]()))return;_0x48cbb4=_0x48cbb4[_0x42de5a(0x1d4)](/\\I\[(\d+)\]/gi,''),_0x48cbb4=_0x48cbb4[_0x42de5a(0x1d4)](/\x1bI\[(\d+)\]/gi,'');if(_0x48cbb4[_0x42de5a(0x2fe)]<=0x0)return;if(_0x48cbb4[_0x42de5a(0x1f2)](/-----/i))return;_0x4cf89c[_0x42de5a(0x3ec)](_0x48cbb4);}else _0x3e4bd7[_0x42de5a(0x1f2)](_0x5c9e19[_0x42de5a(0x2ea)])&&(_0x3e4bd7=_0x3e4bd7[_0x42de5a(0x1d4)](_0x5c9e19[_0x42de5a(0x2ea)],_0x5c9e19[_0x42de5a(0x323)]),_0x3e4bd7=this[_0x42de5a(0x3f0)](_0x3e4bd7));}return _0x3e4bd7;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x409)]=function(_0x4d62a2){const _0x3c9daf=_0x22ff2b;for(const _0x1bf463 of VisuMZ[_0x3c9daf(0x3d5)][_0x3c9daf(0x236)][_0x3c9daf(0x2a9)]){if(_0x3c9daf(0x39c)!==_0x3c9daf(0x20b))_0x4d62a2['match'](_0x1bf463['textCodeCheck'])&&(_0x4d62a2=_0x4d62a2[_0x3c9daf(0x1d4)](_0x1bf463[_0x3c9daf(0x2ea)],_0x1bf463[_0x3c9daf(0x323)]['bind'](this)),_0x4d62a2=this[_0x3c9daf(0x3f0)](_0x4d62a2));else return _0x28f571[_0x3c9daf(0x1b9)]+=_0x17e02a[0x0][_0x3c9daf(0x2fe)],_0x10bcb3(_0x40cc25[0x0][_0x3c9daf(0x412)](0x1,_0x3cb5d3[0x0][_0x3c9daf(0x2fe)]-0x1));}return _0x4d62a2;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x34f)]=function(_0x18f40c){const _0x5f2f7d=_0x22ff2b,_0x59aaf4=_0x18f40c>=0x1?$gameActors[_0x5f2f7d(0x2c5)](_0x18f40c):null,_0x290dce=_0x59aaf4?_0x59aaf4[_0x5f2f7d(0x36d)]():'',_0x3df631=Number(VisuMZ[_0x5f2f7d(0x3d5)][_0x5f2f7d(0x236)][_0x5f2f7d(0x410)]['Actors']);if(this[_0x5f2f7d(0x223)]()&&_0x3df631!==0x0){if(_0x5f2f7d(0x414)===_0x5f2f7d(0x414))return'\x1bC[%1]%2\x1bPREVCOLOR[0]'[_0x5f2f7d(0x197)](_0x3df631,_0x290dce);else _0x252bee[_0x5f2f7d(0x3d5)][_0x5f2f7d(0x25f)][_0x5f2f7d(0x282)](this),this['updateMessageCommonEvents']();}else{if(_0x5f2f7d(0x1a6)!==_0x5f2f7d(0x1a6)){_0x2880b5[_0x5f2f7d(0x3d5)][_0x5f2f7d(0x276)][_0x5f2f7d(0x282)](this,_0xefe693);const _0x5a792a=_0x33e96e[_0x5f2f7d(0x3d5)][_0x5f2f7d(0x236)][_0x5f2f7d(0x410)];_0x528cd5[_0x5f2f7d(0x3d5)][_0x5f2f7d(0x3c3)](_0x4868c2,_0x5a792a[_0x5f2f7d(0x3fe)]);}else return _0x290dce;}},Window_Base['prototype'][_0x22ff2b(0x294)]=function(_0x27085e){const _0x213172=_0x22ff2b,_0xdb7cc2=_0x27085e>=0x1?$gameParty[_0x213172(0x235)]()[_0x27085e-0x1]:null,_0x16add2=_0xdb7cc2?_0xdb7cc2['name']():'',_0x435978=Number(VisuMZ[_0x213172(0x3d5)]['Settings'][_0x213172(0x410)][_0x213172(0x420)]);if(this[_0x213172(0x223)]()&&_0x435978!==0x0)return _0x213172(0x29c)===_0x213172(0x29c)?_0x213172(0x378)[_0x213172(0x197)](_0x435978,_0x16add2):_0x1c02c9[_0x213172(0x3d5)]['Window_ChoiceList_windowX'][_0x213172(0x282)](this);else{if(_0x213172(0x3ab)==='bijey')return _0x16add2;else this[_0x213172(0x226)]=null;}},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x3b9)]=function(_0x122adc){const _0x4df209=_0x22ff2b;if(this[_0x4df209(0x223)]()){if(_0x4df209(0x2ba)!==_0x4df209(0x411))_0x122adc=this[_0x4df209(0x390)](_0x122adc),_0x122adc=this[_0x4df209(0x2af)](_0x122adc);else{if(_0x30b31d[_0x4df209(0x376)]())return![];return this['prepareShowTextCommand'](_0x5a8317),this['addContinuousShowTextCommands'](_0x134200),this[_0x4df209(0x388)](_0x589fe0),this['setWaitMode']('message'),!![];}}return _0x122adc;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x390)]=function(_0x1759e4){const _0x179ddc=_0x22ff2b;for(autoColor of VisuMZ['MessageCore'][_0x179ddc(0x3d2)]){if('Xlqsa'===_0x179ddc(0x2d1)){const _0x194feb=_0x50ffb9[_0x179ddc(0x2cb)](_0x509823['width'],_0x4c2b3e['getMessageWindowWidth']()),_0x53ec0b=_0x10396a[_0x179ddc(0x3ce)](),_0x5461cf=this['calcWindowHeight'](_0x53ec0b,![]),_0x4e6bb0=(_0x4ddd08[_0x179ddc(0x26f)]-_0x194feb)/0x2,_0x4ed373=0x0;return new _0x1fd664(_0x4e6bb0,_0x4ed373,_0x194feb,_0x5461cf);}else _0x1759e4=_0x1759e4[_0x179ddc(0x1d4)](autoColor[0x0],autoColor[0x1]);}return _0x1759e4;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x301)]=function(){const _0x2fb030=_0x22ff2b;this[_0x2fb030(0x308)]=[];},Window_Base['prototype']['registerActorNameAutoColorChanges']=function(){const _0x23aefa=_0x22ff2b;this[_0x23aefa(0x301)]();const _0x112c1d=VisuMZ[_0x23aefa(0x3d5)][_0x23aefa(0x236)][_0x23aefa(0x410)],_0x4c0697=_0x112c1d['Actors'];if(_0x4c0697<=0x0)return;for(const _0x35ca89 of $gameActors['_data']){if('pcppq'===_0x23aefa(0x24d))return _0x3f8d6b;else{if(!_0x35ca89)continue;const _0x218b22=_0x35ca89['name']();if(_0x218b22[_0x23aefa(0x415)]()['length']<=0x0)continue;if(/^\d+$/[_0x23aefa(0x41f)](_0x218b22))continue;if(_0x218b22[_0x23aefa(0x1f2)](/-----/i))continue;let _0x287cdb=VisuMZ[_0x23aefa(0x3d5)][_0x23aefa(0x1d0)](_0x218b22);const _0x3a79bb=new RegExp('\x5cb'+_0x287cdb+'\x5cb','g'),_0x59093a=_0x23aefa(0x378)['format'](_0x4c0697,_0x218b22);this['_autoColorActorNames'][_0x23aefa(0x3ec)]([_0x3a79bb,_0x59093a]);}}},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x2af)]=function(_0x45a0ba){const _0x278df2=_0x22ff2b;this[_0x278df2(0x308)]===undefined&&(_0x278df2(0x329)!==_0x278df2(0x329)?(_0x46934f[_0x278df2(0x3d5)]['Game_Party_gainItem'][_0x278df2(0x282)](this,_0x30adb1,_0x38dd44,_0x17d148),_0x4033d5>0x0&&this['setLastGainedItemData'](_0xe791fc,_0x4bcdce)):this['registerActorNameAutoColorChanges']());for(autoColor of this[_0x278df2(0x308)]){_0x278df2(0x25c)!==_0x278df2(0x25c)?this[_0x278df2(0x36b)](_0x2536f4,_0x37ec7b['clamp'](0x1,0xb)):_0x45a0ba=_0x45a0ba[_0x278df2(0x1d4)](autoColor[0x0],autoColor[0x1]);}return _0x45a0ba;},Window_Base['prototype'][_0x22ff2b(0x1e5)]=function(_0x58bb9e,_0x384881,_0xc4e4c2){const _0x4200f3=_0x22ff2b;if(!_0x58bb9e)return'';const _0x1ca191=_0x58bb9e[_0x384881];let _0x16df6e='';if(_0x1ca191&&_0xc4e4c2&&_0x1ca191[_0x4200f3(0x2be)]){const _0x5bae75=_0x4200f3(0x324);_0x16df6e=_0x5bae75[_0x4200f3(0x197)](_0x1ca191[_0x4200f3(0x2be)],_0x1ca191[_0x4200f3(0x36d)]);}else{if(_0x1ca191){if(_0x4200f3(0x1a1)!==_0x4200f3(0x1a1)){this[_0x4200f3(0x2ab)](_0x19355d);if(this[_0x4200f3(0x247)]())return;_0x264647['drawing']&&(this['_textColorStack']=this[_0x4200f3(0x2d2)]||[],this[_0x4200f3(0x1d3)][_0x4200f3(0x3db)]=this[_0x4200f3(0x2d2)][_0x4200f3(0x3bf)]()||_0x31c203[_0x4200f3(0x309)]());}else _0x16df6e=_0x1ca191[_0x4200f3(0x36d)];}else _0x16df6e='';}return this['isAutoColorAffected']()&&(_0x16df6e=this['applyDatabaseAutoColor'](_0x16df6e,_0x58bb9e)),_0x16df6e;},Window_Base[_0x22ff2b(0x3d4)]['lastGainedObjectName']=function(_0x43ec85){const _0x1c64d4=_0x22ff2b,_0x355948=$gameParty[_0x1c64d4(0x3a4)]();if(_0x355948['id']<0x0)return'';let _0x208a61=null;if(_0x355948[_0x1c64d4(0x3ef)]===0x0)_0x208a61=$dataItems[_0x355948['id']];if(_0x355948['type']===0x1)_0x208a61=$dataWeapons[_0x355948['id']];if(_0x355948['type']===0x2)_0x208a61=$dataArmors[_0x355948['id']];if(!_0x208a61)return'';return _0x43ec85?_0x1c64d4(0x324)[_0x1c64d4(0x197)](_0x208a61['iconIndex'],_0x208a61[_0x1c64d4(0x36d)]):_0x208a61[_0x1c64d4(0x36d)];},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x35e)]=function(){const _0x4b22fa=_0x22ff2b,_0x1b9fb3=$gameParty[_0x4b22fa(0x3a4)]();if(_0x1b9fb3['id']<=0x0)return'';return _0x1b9fb3[_0x4b22fa(0x1c4)];},Window_Base[_0x22ff2b(0x3d4)]['applyDatabaseAutoColor']=function(_0x36a5a0,_0x2163ab){const _0x5064de=_0x22ff2b,_0x2d4de0=VisuMZ[_0x5064de(0x3d5)][_0x5064de(0x236)][_0x5064de(0x410)];let _0x402907=0x0;if(_0x2163ab===$dataActors)_0x402907=_0x2d4de0[_0x5064de(0x420)];if(_0x2163ab===$dataClasses)_0x402907=_0x2d4de0['Classes'];if(_0x2163ab===$dataSkills)_0x402907=_0x2d4de0[_0x5064de(0x196)];if(_0x2163ab===$dataItems)_0x402907=_0x2d4de0[_0x5064de(0x1ca)];if(_0x2163ab===$dataWeapons)_0x402907=_0x2d4de0[_0x5064de(0x1fe)];if(_0x2163ab===$dataArmors)_0x402907=_0x2d4de0[_0x5064de(0x3ae)];if(_0x2163ab===$dataEnemies)_0x402907=_0x2d4de0[_0x5064de(0x3fe)];if(_0x2163ab===$dataStates)_0x402907=_0x2d4de0[_0x5064de(0x3d3)];if(_0x402907>0x0){if(_0x5064de(0x1b8)!==_0x5064de(0x1b8)){const _0x291fb9=this[_0x5064de(0x318)](_0x30a40e)[_0x5064de(0x2e1)](',');if(!_0x377adb[_0x5064de(0x3f6)])return;const _0x458498=_0x291fb9[0x0][_0x5064de(0x415)](),_0x52e4cf=_0x291fb9[0x1]||0x0,_0x204118=_0x291fb9[0x2]||0x0,_0x5136dc=_0x1b9c77[_0x5064de(0x3d6)](_0x458498),_0x3ee33f=this[_0x5064de(0x1d3)][_0x5064de(0x374)];_0x5136dc[_0x5064de(0x3cd)](this['drawBackPicture']['bind'](this,_0x5136dc,_0x4380a1['x'],_0x49fe4f['y'],_0x52e4cf,_0x204118,_0x3ee33f));}else _0x36a5a0=_0x5064de(0x378)[_0x5064de(0x197)](_0x402907,_0x36a5a0);}return _0x36a5a0;},Window_Base[_0x22ff2b(0x3d4)]['prepareWordWrapEscapeCharacters']=function(_0x30da5b){const _0x3674fd=_0x22ff2b;_0x30da5b=_0x30da5b[_0x3674fd(0x1d4)](/<(?:WORDWRAP|WORD WRAP)>/gi,(_0x4e5c08,_0x3f7db8)=>this[_0x3674fd(0x3af)](!![])),_0x30da5b=_0x30da5b[_0x3674fd(0x1d4)](/<(?:NOWORDWRAP|NO WORD WRAP)>/gi,(_0xd3ca6e,_0x203c07)=>this[_0x3674fd(0x3af)](![])),_0x30da5b=_0x30da5b[_0x3674fd(0x1d4)](/<\/(?:WORDWRAP|WORD WRAP)>/gi,(_0x435287,_0x1db8a7)=>this[_0x3674fd(0x3af)](![]));if(_0x30da5b[_0x3674fd(0x1f2)](Window_Message[_0x3674fd(0x401)]))'NPmem'==='NPmem'?this[_0x3674fd(0x3af)](![]):!_0x4f4b0d[_0x3674fd(0x3f6)]?_0x4b6f50[_0x3674fd(0x3d4)][_0x3674fd(0x385)][_0x3674fd(0x282)](this,_0x43865a,_0x141104):_0x15713b[_0x3674fd(0x3d5)]['Window_Message_processEscapeCharacter'][_0x3674fd(0x282)](this,_0x4691ec,_0x53c506);else{if(_0x30da5b[_0x3674fd(0x1f2)](Window_Message[_0x3674fd(0x263)])){if(_0x3674fd(0x3a2)!==_0x3674fd(0x3a2)){_0x4fef75['MessageCore'][_0x3674fd(0x1bd)](_0x3674fd(0x2ec));for(const _0x55da17 of _0x49a1be[_0x3674fd(0x3d5)][_0x3674fd(0x236)][_0x3674fd(0x2ec)]){_0x55da17[_0x3674fd(0x27e)]=_0x55da17[_0x3674fd(0x27e)][_0x3674fd(0x2a4)](),_0x55da17[_0x3674fd(0x2ea)]=new _0x2f8eb4('\x1b'+_0x55da17['Match'],'gi'),_0x55da17[_0x3674fd(0x323)]='\x1b'+_0x55da17['Match'];if(_0x55da17[_0x3674fd(0x1b1)]==='')_0x55da17[_0x3674fd(0x323)]+='[0]';}}else this[_0x3674fd(0x3af)](![]);}}if(!this[_0x3674fd(0x375)]())return _0x30da5b;if(_0x30da5b[_0x3674fd(0x2fe)]<=0x0)return _0x30da5b;if(VisuMZ['MessageCore']['Settings']['WordWrap'][_0x3674fd(0x216)]){if(_0x3674fd(0x1be)==='Ipeae'){var _0x315f61=_0x3b3c81['parameters'][0x1]+_0x1adf70;this[_0x3674fd(0x347)][_0xfc5980][_0x3674fd(0x22c)][0x1]=_0x315f61;}else _0x30da5b=_0x30da5b[_0x3674fd(0x1d4)](/[\n\r]+/g,'\x20'),_0x30da5b=_0x30da5b['replace'](/<(?:BR|LINEBREAK)>/gi,'\x20\x0a');}else _0x3674fd(0x402)!=='ptHzn'?this[_0x3674fd(0x350)](_0xd5dd3,_0x1d6fff):(_0x30da5b=_0x30da5b['replace'](/[\n\r]+/g,''),_0x30da5b=_0x30da5b[_0x3674fd(0x1d4)](/<(?:BR|LINEBREAK)>/gi,'\x0a'));return _0x30da5b=this[_0x3674fd(0x29a)](_0x30da5b),_0x30da5b=_0x30da5b[_0x3674fd(0x2e1)]('\x20')[_0x3674fd(0x1cb)](_0x3674fd(0x199)),_0x30da5b=_0x30da5b[_0x3674fd(0x1d4)](/<(?:BR|LINEBREAK)>/gi,'\x0a'),_0x30da5b=_0x30da5b[_0x3674fd(0x1d4)](/<LINE\x1bWrapBreak[0]BREAK>/gi,'\x0a'),_0x30da5b;},Window_Base[_0x22ff2b(0x3d4)]['addWrapBreakAfterPunctuation']=function(_0x2a45f3){return _0x2a45f3;},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x2b5)]=Window_Base['prototype']['processNewLine'],Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x2ce)]=function(_0x48d98e){const _0x2d9de1=_0x22ff2b;VisuMZ[_0x2d9de1(0x3d5)][_0x2d9de1(0x2b5)][_0x2d9de1(0x282)](this,_0x48d98e),this[_0x2d9de1(0x2c3)](_0x48d98e);},VisuMZ['MessageCore'][_0x22ff2b(0x31a)]=Window_Base[_0x22ff2b(0x3d4)]['processControlCharacter'],Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x354)]=function(_0x320fc0,_0x8c2764){const _0x21b4c1=_0x22ff2b;VisuMZ[_0x21b4c1(0x3d5)]['Window_Base_processControlCharacter'][_0x21b4c1(0x282)](this,_0x320fc0,_0x8c2764),_0x8c2764==='\x1bWrapBreak[0]'&&this[_0x21b4c1(0x325)](_0x320fc0);},Window_Base['prototype'][_0x22ff2b(0x318)]=function(_0x10da39){const _0x3577a4=_0x22ff2b;var _0x18ea9c=/^\<(.*?)\>/['exec'](_0x10da39[_0x3577a4(0x234)][_0x3577a4(0x412)](_0x10da39[_0x3577a4(0x1b9)]));if(_0x18ea9c){if(_0x3577a4(0x269)==='dzlXf')return _0x10da39[_0x3577a4(0x1b9)]+=_0x18ea9c[0x0][_0x3577a4(0x2fe)],String(_0x18ea9c[0x0][_0x3577a4(0x412)](0x1,_0x18ea9c[0x0][_0x3577a4(0x2fe)]-0x1));else this[_0x3577a4(0x260)]();}else{if(_0x3577a4(0x3d0)===_0x3577a4(0x296)){let _0x1e186a=_0x24a890;return _0x1e186a=_0x1e186a[_0x3577a4(0x1d4)](/<(?:BR|LINEBREAK)>/gi,'\x0a'),_0x1e186a=_0x1e186a[_0x3577a4(0x1d4)](/<LINE\x1bWrapBreak[0]BREAK>/gi,'\x0a'),_0x1e186a;}else return'';}},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x217)]=Window_Base['prototype'][_0x22ff2b(0x385)],Window_Base['prototype'][_0x22ff2b(0x385)]=function(_0x2dd3ff,_0x16156e){const _0x47af7b=_0x22ff2b;switch(_0x2dd3ff){case'C':_0x16156e[_0x47af7b(0x3f6)]?VisuMZ['MessageCore']['Window_Base_processEscapeCharacter'][_0x47af7b(0x282)](this,_0x2dd3ff,_0x16156e):this[_0x47af7b(0x2ab)](_0x16156e);break;case'I':case'{':case'}':VisuMZ['MessageCore'][_0x47af7b(0x217)]['call'](this,_0x2dd3ff,_0x16156e);break;case'FS':this[_0x47af7b(0x3f8)](_0x16156e);break;case'PX':this[_0x47af7b(0x211)](_0x16156e);break;case'PY':this[_0x47af7b(0x331)](_0x16156e);break;case _0x47af7b(0x3bd):this['processFontChangeBold'](this['obtainEscapeParam'](_0x16156e));break;case _0x47af7b(0x3a1):this['processDrawCenteredPicture'](_0x16156e);break;case _0x47af7b(0x2f4):this[_0x47af7b(0x32e)](_0x16156e);break;case'COMMONEVENT':this[_0x47af7b(0x241)](_0x16156e);break;case _0x47af7b(0x1d6):this[_0x47af7b(0x1a8)](this[_0x47af7b(0x2ab)](_0x16156e));break;case _0x47af7b(0x21b):this[_0x47af7b(0x2aa)](_0x16156e);break;case _0x47af7b(0x28b):this[_0x47af7b(0x2bb)](_0x16156e);break;case'TEXTALIGNMENT':this[_0x47af7b(0x292)](_0x16156e);break;case'WAIT':this['processCustomWait'](_0x16156e);break;case _0x47af7b(0x2cc):this[_0x47af7b(0x325)](_0x16156e);break;default:this[_0x47af7b(0x1ab)](_0x2dd3ff,_0x16156e);}},Window_Base['prototype'][_0x22ff2b(0x1ab)]=function(_0x5368ce,_0x23e1c2){const _0x2a21b1=_0x22ff2b;for(const _0x462083 of VisuMZ[_0x2a21b1(0x3d5)]['Settings'][_0x2a21b1(0x2ec)]){if(_0x462083[_0x2a21b1(0x27e)]===_0x5368ce){if(_0x462083['Type']==='')this[_0x2a21b1(0x2ab)](_0x23e1c2);_0x462083[_0x2a21b1(0x3e6)]['call'](this,_0x23e1c2);if(this[_0x2a21b1(0x254)]===Window_Message){const _0xba5c4c=_0x462083[_0x2a21b1(0x23e)]||0x0;if(_0xba5c4c>0x0)this[_0x2a21b1(0x1f0)](_0xba5c4c);}}}},Window_Base[_0x22ff2b(0x3d4)]['makeFontBigger']=function(){const _0x4c8b1c=_0x22ff2b;this[_0x4c8b1c(0x1d3)][_0x4c8b1c(0x3ff)]+=VisuMZ['MessageCore'][_0x4c8b1c(0x236)][_0x4c8b1c(0x2da)]['FontChangeValue'],this[_0x4c8b1c(0x1d3)][_0x4c8b1c(0x3ff)]=Math[_0x4c8b1c(0x2cb)](this[_0x4c8b1c(0x1d3)][_0x4c8b1c(0x3ff)],VisuMZ[_0x4c8b1c(0x3d5)][_0x4c8b1c(0x236)]['General']['FontBiggerCap']);},Window_Base['prototype'][_0x22ff2b(0x3ad)]=function(){const _0x113566=_0x22ff2b;this[_0x113566(0x1d3)][_0x113566(0x3ff)]-=VisuMZ[_0x113566(0x3d5)]['Settings'][_0x113566(0x2da)]['FontChangeValue'],this['contents']['fontSize']=Math['max'](this['contents'][_0x113566(0x3ff)],VisuMZ[_0x113566(0x3d5)]['Settings'][_0x113566(0x2da)][_0x113566(0x1fb)]);},Window_Base[_0x22ff2b(0x3d4)]['processFsTextCode']=function(_0xb8084c){const _0x20aac6=_0x22ff2b,_0x3c4a7a=this[_0x20aac6(0x2ab)](_0xb8084c);this[_0x20aac6(0x1d3)][_0x20aac6(0x3ff)]=_0x3c4a7a['clamp'](VisuMZ[_0x20aac6(0x3d5)]['Settings'][_0x20aac6(0x2da)]['FontSmallerCap'],VisuMZ[_0x20aac6(0x3d5)][_0x20aac6(0x236)][_0x20aac6(0x2da)][_0x20aac6(0x26b)]);},Window_Base['prototype'][_0x22ff2b(0x267)]=function(_0x470fb5){const _0x5abdcc=_0x22ff2b;let _0x537a8b=this[_0x5abdcc(0x1d3)][_0x5abdcc(0x3ff)];const _0x3294b0=/\x1b({|}|FS)(\[(\d+)])?/gi;for(;;){if('RLeOt'!==_0x5abdcc(0x265)){if(this[_0x5abdcc(0x1ad)]===_0x481c70)this[_0x5abdcc(0x35d)]();if(this['_MessageCoreSettings'][_0x5abdcc(0x1da)]===_0xda492f)this[_0x5abdcc(0x35d)]();this['_MessageCoreSettings'][_0x5abdcc(0x1da)]=_0x30166f||0x1;}else{const _0x295a6e=_0x3294b0[_0x5abdcc(0x36e)](_0x470fb5);if(!_0x295a6e){if(_0x5abdcc(0x3f3)!==_0x5abdcc(0x3f3))_0x6b79bd=_0x5abdcc(0x378)[_0x5abdcc(0x197)](_0x24248d,_0x5a281f);else break;}const _0x2e42a7=String(_0x295a6e[0x1])[_0x5abdcc(0x2a4)]();if(_0x2e42a7==='{')this[_0x5abdcc(0x232)]();else{if(_0x2e42a7==='}')_0x5abdcc(0x2d4)===_0x5abdcc(0x2d4)?this[_0x5abdcc(0x3ad)]():(this[_0x5abdcc(0x2e7)](_0x5019af),this[_0x5abdcc(0x3e3)](_0x5e5c39),_0x361862[_0x5abdcc(0x3d5)][_0x5abdcc(0x255)]['call'](this,_0x588508),this[_0x5abdcc(0x367)]());else _0x2e42a7==='FS'&&(this[_0x5abdcc(0x1d3)][_0x5abdcc(0x3ff)]=parseInt(_0x295a6e[0x3])[_0x5abdcc(0x3a7)](VisuMZ[_0x5abdcc(0x3d5)]['Settings'][_0x5abdcc(0x2da)][_0x5abdcc(0x1fb)],VisuMZ[_0x5abdcc(0x3d5)][_0x5abdcc(0x236)][_0x5abdcc(0x2da)][_0x5abdcc(0x26b)]));}this['contents']['fontSize']>_0x537a8b&&(_0x537a8b=this['contents'][_0x5abdcc(0x3ff)]);}}return _0x537a8b;},Window_Base['prototype'][_0x22ff2b(0x211)]=function(_0x47c27c){const _0x4b815d=_0x22ff2b;_0x47c27c['x']=this[_0x4b815d(0x2ab)](_0x47c27c),VisuMZ[_0x4b815d(0x3d5)][_0x4b815d(0x236)]['General']['RelativePXPY']&&(_0x47c27c['x']+=_0x47c27c['startX']);},Window_Base['prototype'][_0x22ff2b(0x331)]=function(_0x479221){const _0x12ad52=_0x22ff2b;_0x479221['y']=this[_0x12ad52(0x2ab)](_0x479221),VisuMZ[_0x12ad52(0x3d5)][_0x12ad52(0x236)][_0x12ad52(0x2da)][_0x12ad52(0x222)]&&(_0x479221['y']+=_0x479221[_0x12ad52(0x2b2)]);},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x228)]=function(_0x477dca){const _0xb2f03f=_0x22ff2b;this[_0xb2f03f(0x1d3)][_0xb2f03f(0x316)]=!!_0x477dca;},Window_Base[_0x22ff2b(0x3d4)]['processFontChangeItalic']=function(_0x10b121){const _0x41c270=_0x22ff2b;this[_0x41c270(0x1d3)][_0x41c270(0x243)]=!!_0x10b121;},Window_Base[_0x22ff2b(0x3d4)]['processTextAlignmentChange']=function(_0x49279a){const _0x5c790e=_0x22ff2b,_0x2e6b6c=this['obtainEscapeParam'](_0x49279a);if(!_0x49279a['drawing'])return;switch(_0x2e6b6c){case 0x0:this[_0x5c790e(0x31f)](_0x5c790e(0x1d1));return;case 0x1:this['setTextAlignment'](_0x5c790e(0x39d));break;case 0x2:this[_0x5c790e(0x31f)](_0x5c790e(0x19a));break;case 0x3:this['setTextAlignment'](_0x5c790e(0x3b0));break;}this[_0x5c790e(0x2c3)](_0x49279a);},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x2c3)]=function(_0x55f655){const _0x39cc07=_0x22ff2b;if(!_0x55f655[_0x39cc07(0x3f6)])return;if(_0x55f655['rtl'])return;if(this[_0x39cc07(0x34c)]()==='default')return;let _0x55f342=_0x55f655[_0x39cc07(0x234)][_0x39cc07(0x24b)](_0x39cc07(0x306),_0x55f655[_0x39cc07(0x1b9)]+0x1),_0x1adf80=_0x55f655['text']['indexOf']('\x0a',_0x55f655[_0x39cc07(0x1b9)]+0x1);if(_0x55f342<0x0)_0x55f342=_0x55f655['text'][_0x39cc07(0x2fe)]+0x1;if(_0x1adf80>0x0)_0x55f342=Math[_0x39cc07(0x2cb)](_0x55f342,_0x1adf80);const _0xcd84be=_0x55f655['text'][_0x39cc07(0x2c9)](_0x55f655[_0x39cc07(0x1b9)],_0x55f342),_0x37de6a=this[_0x39cc07(0x1b0)](_0xcd84be)[_0x39cc07(0x244)],_0x5b6e01=_0x55f655['width']||this[_0x39cc07(0x371)]-0x8,_0x3facc0=this[_0x39cc07(0x254)]===Window_Message&&$gameMessage[_0x39cc07(0x366)]()!=='';switch(this[_0x39cc07(0x34c)]()){case _0x39cc07(0x39d):_0x55f655['x']=_0x55f655['startX'];break;case _0x39cc07(0x19a):_0x55f655['x']=_0x55f655[_0x39cc07(0x3ee)],_0x55f655['x']+=Math[_0x39cc07(0x2db)]((_0x5b6e01-_0x37de6a)/0x2);_0x3facc0&&(_0x55f655['x']-=_0x55f655[_0x39cc07(0x3ee)]/0x2);break;case _0x39cc07(0x3b0):_0x55f655['x']=_0x5b6e01-_0x37de6a+_0x55f655[_0x39cc07(0x3ee)];_0x3facc0&&(_0x55f655['x']-=_0x55f655[_0x39cc07(0x3ee)]);break;}},Window_Base[_0x22ff2b(0x3d4)]['textSizeExTextAlignment']=function(_0x5b4d93){const _0x5f1f09=_0x22ff2b;_0x5b4d93=_0x5b4d93[_0x5f1f09(0x1d4)](/\x1b!/g,''),_0x5b4d93=_0x5b4d93[_0x5f1f09(0x1d4)](/\x1b\|/g,''),_0x5b4d93=_0x5b4d93['replace'](/\x1b\./g,'');const _0x574b21=this[_0x5f1f09(0x27d)](_0x5b4d93,0x0,0x0,0x0),_0x10f1d7=this['getPreservedFontSettings']();return _0x574b21[_0x5f1f09(0x3f6)]=![],this['processAllText'](_0x574b21),this['returnPreservedFontSettings'](_0x10f1d7),{'width':_0x574b21['outputWidth'],'height':_0x574b21[_0x5f1f09(0x273)]};},Window_Base[_0x22ff2b(0x29d)]=VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x236)][_0x22ff2b(0x416)][_0x22ff2b(0x40e)]||0x0,Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x325)]=function(_0x5543d1){const _0x15ee02=_0x22ff2b,_0x387d4f=(_0x5543d1['rtl']?-0x1:0x1)*this[_0x15ee02(0x271)]('\x20');_0x5543d1['x']+=_0x387d4f;if(this[_0x15ee02(0x2ab)](_0x5543d1)>0x0)_0x5543d1['x']+=_0x387d4f;if(_0x5543d1[_0x15ee02(0x22e)])return;let _0x33263b=_0x5543d1[_0x15ee02(0x234)][_0x15ee02(0x24b)]('\x1bWrapBreak[0]',_0x5543d1[_0x15ee02(0x1b9)]+0x1),_0x30213d=_0x5543d1[_0x15ee02(0x234)]['indexOf']('\x0a',_0x5543d1[_0x15ee02(0x1b9)]+0x1);if(_0x33263b<0x0)_0x33263b=_0x5543d1[_0x15ee02(0x234)][_0x15ee02(0x2fe)]+0x1;if(_0x30213d>0x0)_0x33263b=Math[_0x15ee02(0x2cb)](_0x33263b,_0x30213d);const _0x29d06a=_0x5543d1[_0x15ee02(0x234)][_0x15ee02(0x2c9)](_0x5543d1['index'],_0x33263b),_0x166bb6=this[_0x15ee02(0x225)](_0x29d06a)[_0x15ee02(0x244)];let _0x35a972=_0x5543d1[_0x15ee02(0x244)]||this['innerWidth'];_0x35a972-=Window_Base['WORD_WRAP_PADDING'];if(this[_0x15ee02(0x254)]===Window_Message){if(_0x15ee02(0x26a)!==_0x15ee02(0x26a)){if(this[_0x15ee02(0x247)]())return;_0x1e0a6a=_0x14a9b0[_0x15ee02(0x1d4)](/\,/g,''),this[_0x15ee02(0x2d2)]=this[_0x15ee02(0x2d2)]||[],this[_0x15ee02(0x2d2)][_0x15ee02(0x34e)](this[_0x15ee02(0x1d3)][_0x15ee02(0x3db)]),_0x1f4ddf[_0x15ee02(0x3d5)]['Window_Base_changeTextColor'][_0x15ee02(0x282)](this,_0x4f534f);}else{const _0x3a701b=$gameMessage[_0x15ee02(0x366)]()===''?0x0:ImageManager['faceWidth']+0x14;_0x35a972-=_0x3a701b,VisuMZ[_0x15ee02(0x3d5)][_0x15ee02(0x236)][_0x15ee02(0x416)][_0x15ee02(0x3ea)]&&(_0x35a972-=_0x3a701b);}}let _0x1dbe36=![];if(_0x5543d1['x']+_0x166bb6>_0x5543d1[_0x15ee02(0x3ee)]+_0x35a972)_0x1dbe36=!![];if(_0x166bb6===0x0)_0x1dbe36=!![];_0x1dbe36&&(_0x5543d1[_0x15ee02(0x234)]=_0x5543d1[_0x15ee02(0x234)][_0x15ee02(0x412)](0x0,_0x5543d1[_0x15ee02(0x1b9)])+'\x0a'+_0x5543d1[_0x15ee02(0x234)][_0x15ee02(0x229)](_0x5543d1[_0x15ee02(0x1b9)]));},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x225)]=function(_0x10196b){const _0x32e4bc=_0x22ff2b,_0x3b4f12=this[_0x32e4bc(0x27d)](_0x10196b,0x0,0x0,0x0),_0x1fb0f3=this['getPreservedFontSettings']();return _0x3b4f12['drawing']=![],this[_0x32e4bc(0x3af)](![]),this['processAllText'](_0x3b4f12),this[_0x32e4bc(0x3af)](!![]),this[_0x32e4bc(0x28f)](_0x1fb0f3),{'width':_0x3b4f12['outputWidth'],'height':_0x3b4f12['outputHeight']};},Window_Base['prototype'][_0x22ff2b(0x241)]=function(_0xb45138){const _0x1ae699=_0x22ff2b;return this[_0x1ae699(0x2ab)](_0xb45138);},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x2aa)]=function(_0x5e3bf7){const _0x5108fd=_0x22ff2b,_0x256b6d=this[_0x5108fd(0x318)](_0x5e3bf7)[_0x5108fd(0x2e1)](',');if(!_0x5e3bf7[_0x5108fd(0x3f6)])return;const _0x14d5e1=_0x256b6d[0x0][_0x5108fd(0x415)](),_0x1a6223=_0x256b6d[0x1]||0x0,_0x57d0b9=_0x256b6d[0x2]||0x0,_0x5b6db9=ImageManager['loadPicture'](_0x14d5e1),_0x124c0a=this[_0x5108fd(0x1d3)][_0x5108fd(0x374)];_0x5b6db9[_0x5108fd(0x3cd)](this[_0x5108fd(0x274)][_0x5108fd(0x1ff)](this,_0x5b6db9,_0x5e3bf7['x'],_0x5e3bf7['y'],_0x1a6223,_0x57d0b9,_0x124c0a));},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x274)]=function(_0x3db193,_0x49d4e9,_0x3ef377,_0x2b384f,_0x280c79,_0x3a0567){const _0x57cc32=_0x22ff2b;_0x2b384f=_0x2b384f||_0x3db193['width'],_0x280c79=_0x280c79||_0x3db193[_0x57cc32(0x293)],this[_0x57cc32(0x19c)]['paintOpacity']=_0x3a0567,this[_0x57cc32(0x19c)]['blt'](_0x3db193,0x0,0x0,_0x3db193[_0x57cc32(0x244)],_0x3db193['height'],_0x49d4e9,_0x3ef377,_0x2b384f,_0x280c79),this[_0x57cc32(0x19c)][_0x57cc32(0x374)]=0xff;},Window_Base['prototype'][_0x22ff2b(0x1bf)]=function(_0x1524bd){const _0x52411f=_0x22ff2b,_0x2a0ff3=this[_0x52411f(0x318)](_0x1524bd)[_0x52411f(0x2e1)](',');if(!_0x1524bd[_0x52411f(0x3f6)])return;const _0x1879be=_0x2a0ff3[0x0][_0x52411f(0x415)](),_0x1e01ad=ImageManager[_0x52411f(0x3d6)](_0x1879be),_0x2a6fdc=JsonEx['makeDeepCopy'](_0x1524bd),_0x43ffa6=this[_0x52411f(0x1d3)][_0x52411f(0x374)];_0x1e01ad['addLoadListener'](this['drawBackCenteredPicture'][_0x52411f(0x1ff)](this,_0x1e01ad,_0x2a6fdc,_0x43ffa6));},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x1d9)]=function(_0x2f2434,_0x7440af,_0x38dd4b){const _0x985996=_0x22ff2b,_0x7163bf=_0x7440af[_0x985996(0x244)]||this['innerWidth'],_0x3ab089=this[_0x985996(0x3e2)]!==undefined?this[_0x985996(0x2c6)]():this[_0x985996(0x203)],_0x238f92=_0x7163bf/_0x2f2434[_0x985996(0x244)],_0x1226d6=_0x3ab089/_0x2f2434[_0x985996(0x293)],_0x16a092=Math[_0x985996(0x2cb)](_0x238f92,_0x1226d6,0x1),_0x322004=this['_index']!==undefined?(this[_0x985996(0x1f6)](0x0)['height']-this[_0x985996(0x21f)]())/0x2:0x0,_0x757cbf=_0x2f2434[_0x985996(0x244)]*_0x16a092,_0x1a18a2=_0x2f2434[_0x985996(0x293)]*_0x16a092,_0x23cf57=Math[_0x985996(0x2db)]((_0x7163bf-_0x757cbf)/0x2)+_0x7440af['startX'],_0x48105a=Math[_0x985996(0x2db)]((_0x3ab089-_0x1a18a2)/0x2)+_0x7440af['startY']-_0x322004*0x2;this[_0x985996(0x19c)][_0x985996(0x374)]=_0x38dd4b,this['contentsBack'][_0x985996(0x270)](_0x2f2434,0x0,0x0,_0x2f2434[_0x985996(0x244)],_0x2f2434[_0x985996(0x293)],_0x23cf57,_0x48105a,_0x757cbf,_0x1a18a2),this[_0x985996(0x19c)][_0x985996(0x374)]=0xff;},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x32e)]=function(_0xdedfa){const _0x6eb6eb=_0x22ff2b,_0x88c462=this[_0x6eb6eb(0x2ab)](_0xdedfa);if(_0xdedfa[_0x6eb6eb(0x3f6)])this[_0x6eb6eb(0x32d)](_0x88c462>0x0);},Window_Base[_0x22ff2b(0x3d4)][_0x22ff2b(0x30e)]=function(_0x47c189){const _0x25e89f=_0x22ff2b,_0x4661db=this[_0x25e89f(0x2ab)](_0x47c189);this[_0x25e89f(0x254)]===Window_Message&&_0x47c189[_0x25e89f(0x3f6)]&&this['startWait'](_0x4661db);},Window_Help[_0x22ff2b(0x3d4)]['resetWordWrap']=function(){const _0xf4f194=_0x22ff2b;this[_0xf4f194(0x3af)]($gameSystem[_0xf4f194(0x2b4)]());},Window_Help[_0x22ff2b(0x3d4)]['isAutoColorAffected']=function(){return!![];},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x2fc)]=Window_Help[_0x22ff2b(0x3d4)][_0x22ff2b(0x2b1)],Window_Help[_0x22ff2b(0x3d4)][_0x22ff2b(0x2b1)]=function(){const _0x295603=_0x22ff2b;this[_0x295603(0x301)](),VisuMZ[_0x295603(0x3d5)][_0x295603(0x2fc)][_0x295603(0x282)](this),this[_0x295603(0x33b)]();},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x1d8)]=Window_Options['prototype'][_0x22ff2b(0x34d)],Window_Options[_0x22ff2b(0x3d4)][_0x22ff2b(0x34d)]=function(){const _0x227fcb=_0x22ff2b;VisuMZ['MessageCore'][_0x227fcb(0x1d8)][_0x227fcb(0x282)](this),this['addMessageCoreCommands']();},Window_Options[_0x22ff2b(0x3d4)][_0x22ff2b(0x38c)]=function(){const _0x318ec0=_0x22ff2b;VisuMZ[_0x318ec0(0x3d5)][_0x318ec0(0x236)][_0x318ec0(0x219)][_0x318ec0(0x1e4)]&&(_0x318ec0(0x2a1)!==_0x318ec0(0x2a1)?(_0x467ec7[_0x318ec0(0x3d5)][_0x318ec0(0x2b5)][_0x318ec0(0x282)](this,_0x2caecb),this[_0x318ec0(0x2c3)](_0x5a8640)):this['addMessageCoreTextSpeedCommand']());},Window_Options[_0x22ff2b(0x3d4)][_0x22ff2b(0x40f)]=function(){const _0x94a3e3=_0x22ff2b,_0x5db9ee=TextManager[_0x94a3e3(0x40a)],_0x459abf=_0x94a3e3(0x399);this[_0x94a3e3(0x37f)](_0x5db9ee,_0x459abf);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x2bf)]=Window_Options['prototype'][_0x22ff2b(0x202)],Window_Options['prototype'][_0x22ff2b(0x202)]=function(_0x12011b){const _0x4658bb=_0x22ff2b,_0x4c4678=this[_0x4658bb(0x3dc)](_0x12011b);if(_0x4c4678===_0x4658bb(0x399))return this[_0x4658bb(0x2ac)]();return VisuMZ[_0x4658bb(0x3d5)][_0x4658bb(0x2bf)]['call'](this,_0x12011b);},VisuMZ['MessageCore'][_0x22ff2b(0x1ec)]=Window_Options['prototype'][_0x22ff2b(0x355)],Window_Options[_0x22ff2b(0x3d4)]['isVolumeSymbol']=function(_0x4c24a1){const _0x58106f=_0x22ff2b;if(_0x4c24a1===_0x58106f(0x399))return!![];return VisuMZ[_0x58106f(0x3d5)][_0x58106f(0x1ec)]['call'](this,_0x4c24a1);},Window_Options[_0x22ff2b(0x3d4)][_0x22ff2b(0x2ac)]=function(){const _0x354a79=_0x22ff2b,_0x90656c=this[_0x354a79(0x333)](_0x354a79(0x399));return _0x90656c>0xa?TextManager['instantTextSpeed']:_0x90656c;},VisuMZ['MessageCore']['Window_Options_changeVolume']=Window_Options[_0x22ff2b(0x3d4)][_0x22ff2b(0x19f)],Window_Options[_0x22ff2b(0x3d4)][_0x22ff2b(0x19f)]=function(_0x4e879b,_0x3835f9,_0x301d82){const _0x3e400f=_0x22ff2b;if(_0x4e879b===_0x3e400f(0x399))return this[_0x3e400f(0x3e0)](_0x4e879b,_0x3835f9,_0x301d82);VisuMZ[_0x3e400f(0x3d5)][_0x3e400f(0x198)][_0x3e400f(0x282)](this,_0x4e879b,_0x3835f9,_0x301d82);},Window_Options[_0x22ff2b(0x3d4)]['changeTextSpeed']=function(_0x52d64a,_0x58abda,_0x331924){const _0x2eb1de=_0x22ff2b,_0xf9cfae=this[_0x2eb1de(0x333)](_0x52d64a),_0x18e416=0x1,_0x200c0d=_0xf9cfae+(_0x58abda?_0x18e416:-_0x18e416);_0x200c0d>0xb&&_0x331924?this[_0x2eb1de(0x36b)](_0x52d64a,0x1):this['changeValue'](_0x52d64a,_0x200c0d[_0x2eb1de(0x3a7)](0x1,0xb));},Window_Message['prototype'][_0x22ff2b(0x218)]=function(){const _0x3774fe=_0x22ff2b;let _0x1e760c=Window_Base[_0x3774fe(0x3d4)][_0x3774fe(0x218)]['call'](this);return _0x1e760c-=this[_0x3774fe(0x369)](),_0x1e760c;},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x342)]=function(){const _0x5d2c21=_0x22ff2b;Window_Base[_0x5d2c21(0x3d4)][_0x5d2c21(0x342)][_0x5d2c21(0x282)](this);if(VisuMZ['MessageCore']['Settings'][_0x5d2c21(0x2da)][_0x5d2c21(0x396)]){if(_0x5d2c21(0x32c)!==_0x5d2c21(0x382))this[_0x5d2c21(0x19b)]();else for(const _0x497ebf of _0x144bab[_0x5d2c21(0x3d5)][_0x5d2c21(0x236)]['TextCodeActions']){if(_0x497ebf[_0x5d2c21(0x27e)]===_0x203136){if(_0x497ebf['Type']==='')this['obtainEscapeParam'](_0x3a499b);_0x497ebf[_0x5d2c21(0x3e6)][_0x5d2c21(0x282)](this,_0x3c0120);if(this[_0x5d2c21(0x254)]===_0x1a49bd){const _0x2e7371=_0x497ebf[_0x5d2c21(0x23e)]||0x0;if(_0x2e7371>0x0)this[_0x5d2c21(0x1f0)](_0x2e7371);}}}}},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x19b)]=function(){const _0x14beb0=_0x22ff2b;this['_dimmerSprite']['x']=Math['round'](this[_0x14beb0(0x244)]/0x2),this[_0x14beb0(0x285)][_0x14beb0(0x23a)]['x']=0.5,this[_0x14beb0(0x285)][_0x14beb0(0x2bd)]['x']=Graphics['width'];},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x2a8)]=Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x231)],Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x231)]=function(){const _0x33af54=_0x22ff2b;VisuMZ[_0x33af54(0x3d5)][_0x33af54(0x2a8)][_0x33af54(0x282)](this),this[_0x33af54(0x301)](),this[_0x33af54(0x33b)](),this[_0x33af54(0x32d)](![]),this[_0x33af54(0x31f)](_0x33af54(0x1d1)),this[_0x33af54(0x1cf)](VisuMZ[_0x33af54(0x3d5)][_0x33af54(0x236)]['General'][_0x33af54(0x27b)]);},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x33b)]=function(){const _0x20c95e=_0x22ff2b;this[_0x20c95e(0x3af)]($gameSystem[_0x20c95e(0x272)]());},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x223)]=function(){return!![];},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x1cf)]=function(_0x3731c6){const _0x5e55d5=_0x22ff2b,_0x2ce932=0xb-ConfigManager[_0x5e55d5(0x399)];_0x3731c6=Math[_0x5e55d5(0x278)](_0x3731c6*_0x2ce932),this[_0x5e55d5(0x3de)]=_0x3731c6,this[_0x5e55d5(0x35c)]=_0x3731c6;},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x3f2)]=Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x383)],Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x383)]=function(){const _0x9e3931=_0x22ff2b;return VisuMZ[_0x9e3931(0x3d5)][_0x9e3931(0x3f2)]['call'](this)||Input[_0x9e3931(0x1ba)](VisuMZ[_0x9e3931(0x3d5)][_0x9e3931(0x236)][_0x9e3931(0x2da)]['FastForwardKey']);},VisuMZ[_0x22ff2b(0x3d5)]['Window_Message_updatePlacement']=Window_Message['prototype'][_0x22ff2b(0x19e)],Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x19e)]=function(){const _0x553ef8=_0x22ff2b;let _0x1371e4=this['y'];VisuMZ[_0x553ef8(0x3d5)][_0x553ef8(0x299)]['call'](this);if(this[_0x553ef8(0x25b)])this['y']=_0x1371e4;this[_0x553ef8(0x3c4)](),this[_0x553ef8(0x1f4)]();},VisuMZ[_0x22ff2b(0x3d5)]['Window_Message_newPage']=Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x2ed)],Window_Message['prototype'][_0x22ff2b(0x2ed)]=function(_0x45ba08){const _0x33af5f=_0x22ff2b;this[_0x33af5f(0x2e7)](_0x45ba08),this[_0x33af5f(0x3e3)](_0x45ba08),VisuMZ[_0x33af5f(0x3d5)][_0x33af5f(0x255)][_0x33af5f(0x282)](this,_0x45ba08),this[_0x33af5f(0x367)]();},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x2e7)]=function(_0x3aadf3){const _0x18fbf4=_0x22ff2b;if(!_0x3aadf3)return;_0x3aadf3[_0x18fbf4(0x234)]=this[_0x18fbf4(0x377)](_0x3aadf3[_0x18fbf4(0x234)]),_0x3aadf3['text']=Window_Base[_0x18fbf4(0x3d4)][_0x18fbf4(0x1c2)][_0x18fbf4(0x282)](this,_0x3aadf3[_0x18fbf4(0x234)]);},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x1c2)]=function(_0x33a75c){return _0x33a75c;},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x3e3)]=function(_0x3ebbf7){const _0x4e965b=_0x22ff2b;this[_0x4e965b(0x2d6)](_0x3ebbf7),this['prepareAutoSizeEscapeCharacters'](_0x3ebbf7),this['updateDimensions']();},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x313)]=Window_Message['prototype'][_0x22ff2b(0x3c6)],Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x3c6)]=function(){const _0x2ae7ed=_0x22ff2b;VisuMZ[_0x2ae7ed(0x3d5)]['Window_Message_terminateMessage']['call'](this),this[_0x2ae7ed(0x231)]();if(this[_0x2ae7ed(0x2d0)])this[_0x2ae7ed(0x332)]();},Window_Message['prototype'][_0x22ff2b(0x21d)]=function(){const _0x3ab529=_0x22ff2b;this[_0x3ab529(0x244)]=$gameSystem['getMessageWindowWidth']()+this[_0x3ab529(0x2fd)]();;this[_0x3ab529(0x244)]=Math[_0x3ab529(0x2cb)](Graphics[_0x3ab529(0x244)],this[_0x3ab529(0x244)]);const _0x523c9d=$gameSystem[_0x3ab529(0x3ce)]();this[_0x3ab529(0x293)]=SceneManager[_0x3ab529(0x261)][_0x3ab529(0x2a3)](_0x523c9d,![])+this[_0x3ab529(0x369)](),this['height']=Math[_0x3ab529(0x2cb)](Graphics[_0x3ab529(0x293)],this[_0x3ab529(0x293)]);if($gameTemp['_centerMessageWindow'])this[_0x3ab529(0x302)]();},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x2fd)]=function(){return 0x0;},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x369)]=function(){return 0x0;},Window_Message['prototype'][_0x22ff2b(0x302)]=function(){const _0x43f435=_0x22ff2b;this['x']=(Graphics[_0x43f435(0x26f)]-this['width'])/0x2,$gameTemp[_0x43f435(0x1a4)]=undefined,this[_0x43f435(0x1f4)]();},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x344)]=function(){const _0x3d35fa=_0x22ff2b,_0x46c3a2={'x':this['x'],'y':this['y']};Window_Base['prototype'][_0x3d35fa(0x344)]['call'](this),this[_0x3d35fa(0x379)](_0x46c3a2);},Window_Message['prototype'][_0x22ff2b(0x2dc)]=function(){return!![];},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x379)]=function(_0x2d8311){const _0x3a8fd1=_0x22ff2b;this[_0x3a8fd1(0x391)]&&(this['_nameBoxWindow']['x']+=this['x']-_0x2d8311['x'],this[_0x3a8fd1(0x391)]['y']+=this['y']-_0x2d8311['y']);},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x394)]=function(_0xc3a085,_0xe1d4cd){const _0x2940a6=_0x22ff2b;this[_0x2940a6(0x2d5)](this[_0x2940a6(0x29f)]['x'],this[_0x2940a6(0x387)]*(Graphics['boxHeight']-this[_0x2940a6(0x293)])/0x2,this['_resetRect']['width'],this[_0x2940a6(0x29f)][_0x2940a6(0x293)],_0xc3a085,_0xe1d4cd);},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x241)]=function(_0x2ca45d){const _0x5690f2=_0x22ff2b,_0x1a8d60=Window_Base[_0x5690f2(0x3d4)][_0x5690f2(0x241)][_0x5690f2(0x282)](this,_0x2ca45d);_0x2ca45d[_0x5690f2(0x3f6)]&&this['launchMessageCommonEvent'](_0x1a8d60);},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x1f0)]=function(_0x526ebc){if($gameParty['inBattle']()){}else $gameMap['addMessageCommonEvent'](_0x526ebc);},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x3b7)]=function(_0x55a11b){const _0x923f62=_0x22ff2b;this[_0x923f62(0x3de)]--,this['_textDelayCount']<=0x0&&(this['onProcessCharacter'](_0x55a11b),Window_Base[_0x923f62(0x3d4)]['processCharacter'][_0x923f62(0x282)](this,_0x55a11b));},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x22d)]=function(_0x336bde){const _0x422cf7=_0x22ff2b;this[_0x422cf7(0x3de)]=this[_0x422cf7(0x35c)];if(this[_0x422cf7(0x35c)]<=0x0)this['_showFast']=!![];},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x1dd)]=Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x385)],Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x385)]=function(_0x2ed6bc,_0x25dcaf){const _0x347efc=_0x22ff2b;if(!_0x25dcaf[_0x347efc(0x3f6)]){if(_0x347efc(0x2b9)===_0x347efc(0x2b9))Window_Base[_0x347efc(0x3d4)][_0x347efc(0x385)]['call'](this,_0x2ed6bc,_0x25dcaf);else{const _0x206b2d=_0x4b96c6(_0x30f896['$1']);_0x206b2d!==_0x5746f0[_0x419a4b][_0x347efc(0x38e)]&&(_0x3228be(_0x347efc(0x1f8)['format'](_0x32385b,_0x206b2d)),_0x1c201c[_0x347efc(0x341)]());}}else VisuMZ[_0x347efc(0x3d5)]['Window_Message_processEscapeCharacter'][_0x347efc(0x282)](this,_0x2ed6bc,_0x25dcaf);},Window_Message['prototype'][_0x22ff2b(0x2d6)]=function(_0x20c919){const _0x1a4b4e=_0x22ff2b;let _0x2669e6=_0x20c919[_0x1a4b4e(0x234)];this[_0x1a4b4e(0x30d)]={};if(this['isWordWrapEnabled']())return _0x2669e6;_0x2669e6=_0x2669e6[_0x1a4b4e(0x1d4)](/<POSITION:[ ]*(.*)>/gi,(_0x4bc96d,_0x36be09)=>{const _0x26affb=_0x1a4b4e,_0x51a1cb=_0x36be09['split'](',')['map'](_0x38729f=>Number(_0x38729f)||0x0);if(_0x51a1cb[0x0]!==undefined)this[_0x26affb(0x30d)]['x']=Number(_0x51a1cb[0x0]);if(_0x51a1cb[0x1]!==undefined)this['_forcedPosition']['y']=Number(_0x51a1cb[0x1]);if(_0x51a1cb[0x2]!==undefined)this[_0x26affb(0x30d)][_0x26affb(0x244)]=Number(_0x51a1cb[0x2]);if(_0x51a1cb[0x3]!==undefined)this[_0x26affb(0x30d)][_0x26affb(0x293)]=Number(_0x51a1cb[0x3]);return'';}),_0x2669e6=_0x2669e6[_0x1a4b4e(0x1d4)](/<COORDINATES:[ ]*(.*)>/gi,(_0x583034,_0x10a3b9)=>{const _0x196ec5=_0x1a4b4e,_0x1a5f92=_0x10a3b9['split'](',')['map'](_0x38d724=>Number(_0x38d724)||0x0);if(_0x1a5f92[0x0]!==undefined)this[_0x196ec5(0x30d)]['x']=Number(_0x1a5f92[0x0]);if(_0x1a5f92[0x1]!==undefined)this[_0x196ec5(0x30d)]['y']=Number(_0x1a5f92[0x1]);return'';}),_0x2669e6=_0x2669e6[_0x1a4b4e(0x1d4)](/<DIMENSIONS:[ ]*(.*)>/gi,(_0x5b3489,_0x30ddc6)=>{const _0x3ce7d1=_0x1a4b4e,_0x1de7d6=_0x30ddc6[_0x3ce7d1(0x2e1)](',')[_0x3ce7d1(0x384)](_0x56ed52=>Number(_0x56ed52)||0x0);if(_0x1de7d6[0x0]!==undefined)this['_forcedPosition']['width']=Number(_0x1de7d6[0x2]);if(_0x1de7d6[0x1]!==undefined)this['_forcedPosition']['height']=Number(_0x1de7d6[0x3]);return'';}),_0x20c919[_0x1a4b4e(0x234)]=_0x2669e6;},Window_Message['prototype'][_0x22ff2b(0x3c4)]=function(){const _0x141728=_0x22ff2b;this[_0x141728(0x30d)]=this[_0x141728(0x30d)]||{};const _0x44e41b=['x','y',_0x141728(0x244),'height'];for(const _0x8beca7 of _0x44e41b){if(_0x141728(0x253)==='RzxWO')this[_0x141728(0x30d)][_0x8beca7]!==undefined&&(this[_0x8beca7]=Number(this[_0x141728(0x30d)][_0x8beca7]));else{if(this[_0x141728(0x1ad)]===_0x7b5368)this[_0x141728(0x35d)]();if(this[_0x141728(0x1ad)][_0x141728(0x2f3)]===_0x5dc0dd)this['initMessageCore']();return this[_0x141728(0x1ad)][_0x141728(0x2f3)];}}},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x372)]=function(_0xab4552){const _0x221d8d=_0x22ff2b;let _0x35992f=_0xab4552[_0x221d8d(0x234)];_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTO|AUTOSIZE|AUTO SIZE)>/gi,()=>{const _0x17e050=_0x221d8d;return'qLWYT'===_0x17e050(0x38b)?_0x53a74a[_0x17e050(0x3d4)]['convertTextMacros']['call'](this,_0x2149d7):(this[_0x17e050(0x201)](_0x35992f,!![],!![]),this[_0x17e050(0x233)](_0x17e050(0x266)),'');}),_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTOWIDTH|AUTO WIDTH)>/gi,()=>{const _0x16e73f=_0x221d8d;return this[_0x16e73f(0x201)](_0x35992f,!![],![]),this[_0x16e73f(0x233)](_0x16e73f(0x266)),'';}),_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTOHEIGHT|AUTO HEIGHT)>/gi,()=>{const _0x17598a=_0x221d8d;return this[_0x17598a(0x201)](_0x35992f,![],!![]),this['processAutoPosition'](_0x17598a(0x266)),'';});if(SceneManager[_0x221d8d(0x35a)]())'vUFDf'===_0x221d8d(0x334)?(_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTOACTOR|AUTO ACTOR):[ ](.*?)>/gi,(_0x55f72b,_0x46af06)=>{const _0x2d8f8b=_0x221d8d;return this['processAutoSize'](_0x35992f,!![],!![]),this[_0x2d8f8b(0x233)](_0x2d8f8b(0x364),Number(_0x46af06)||0x1),'';}),_0x35992f=_0x35992f['replace'](/<(?:AUTOPARTY|AUTO PARTY):[ ](.*?)>/gi,(_0x124b2a,_0x555ccd)=>{const _0x3065c3=_0x221d8d;if('pnLyQ'!==_0x3065c3(0x28c))return this[_0x3065c3(0x201)](_0x35992f,!![],!![]),this['processAutoPosition'](_0x3065c3(0x2d9),Number(_0x555ccd)||0x0),'';else{if(_0x572b1b[_0x3065c3(0x290)](_0x9ef048))return!![];}}),_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTOENEMY|AUTO ENEMY):[ ](.*?)>/gi,(_0x3d57b1,_0x5c5aa8)=>{const _0x3fc48a=_0x221d8d;return this[_0x3fc48a(0x201)](_0x35992f,!![],!![]),this['processAutoPosition'](_0x3fc48a(0x2ca),Number(_0x5c5aa8)||0x0),'';})):_0x2e17d9['MessageCore'][_0x221d8d(0x236)][_0x221d8d(0x219)]['AddOption']&&this['addMessageCoreTextSpeedCommand']();else SceneManager[_0x221d8d(0x257)]()&&(_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTOPLAYER|AUTO PLAYER)>/gi,(_0xa1e9f0,_0x905534)=>{const _0x3583a5=_0x221d8d;return this[_0x3583a5(0x201)](_0x35992f,!![],!![]),this[_0x3583a5(0x233)](_0x3583a5(0x37b),0x0),'';}),_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTOACTOR|AUTO ACTOR):[ ](.*?)>/gi,(_0x271fe2,_0x490658)=>{const _0x205371=_0x221d8d;if('KEeYo'===_0x205371(0x370))return this[_0x205371(0x201)](_0x35992f,!![],!![]),this[_0x205371(0x233)](_0x205371(0x2a2),Number(_0x490658)||0x1),'';else{if(!_0x191d81[_0x205371(0x290)](_0x41083d))return![];}}),_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTOPARTY|AUTO PARTY):[ ](.*?)>/gi,(_0x39f947,_0x5e6372)=>{const _0x298ed4=_0x221d8d;return this['processAutoSize'](_0x35992f,!![],!![]),this[_0x298ed4(0x233)](_0x298ed4(0x2e0),Number(_0x5e6372)||0x0),'';}),_0x35992f=_0x35992f[_0x221d8d(0x1d4)](/<(?:AUTOEVENT|AUTO EVENT):[ ](.*?)>/gi,(_0x4633f6,_0x426412)=>{const _0x307afc=_0x221d8d;return this['processAutoSize'](_0x35992f,!![],!![]),this['processAutoPosition'](_0x307afc(0x321),Number(_0x426412)||0x0),'';}));_0xab4552[_0x221d8d(0x234)]=_0x35992f;},Window_Message[_0x22ff2b(0x401)]=/<(?:AUTO|AUTOSIZE|AUTO SIZE|AUTOWIDTH|AUTO WIDTH|AUTOHEIGHT|AUTO HEIGHT|AUTOPLAYER|AUTO PLAYER)>/gi,Window_Message['_autoPosRegExp']=/<(?:AUTOPARTY|AUTO PARTY|AUTOPLAYER|AUTO PLAYER|AUTOEVENT|AUTO EVENT|AUTOENEMY|AUTO ENEMY|AUTOACTOR|AUTO ACTOR):[ ](.*?)>/gi,Window_Message['prototype'][_0x22ff2b(0x201)]=function(_0x44437a,_0x36757c,_0x2650fb){const _0x32ed10=_0x22ff2b;_0x44437a=_0x44437a['replace'](Window_Message['_autoSizeRegexp'],''),_0x44437a=_0x44437a[_0x32ed10(0x1d4)](Window_Message[_0x32ed10(0x263)],''),this['_autoSizeCheck']=!![];const _0x15e71c=this[_0x32ed10(0x37d)](_0x44437a);if(_0x36757c){let _0x52c14a=_0x15e71c[_0x32ed10(0x244)]+$gameSystem[_0x32ed10(0x39a)]()*0x2+0x6;const _0x1ba665=$gameMessage[_0x32ed10(0x366)]()!=='',_0x2ceb93=ImageManager[_0x32ed10(0x3d9)],_0x1e7a87=0x14;_0x52c14a+=_0x1ba665?_0x2ceb93+_0x1e7a87:0x4;if(_0x52c14a%0x2!==0x0)_0x52c14a+=0x1;$gameSystem[_0x32ed10(0x3e7)](_0x52c14a);}if(_0x2650fb){let _0x631b6=Math[_0x32ed10(0x250)](_0x15e71c[_0x32ed10(0x293)]/this[_0x32ed10(0x21f)]());$gameSystem[_0x32ed10(0x40c)](_0x631b6);}this[_0x32ed10(0x2f9)](),this['_autoSizeCheck']=![],this['_messagePositionReset']=!![];},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x2f9)]=function(){const _0x419e7d=_0x22ff2b;this[_0x419e7d(0x21d)](),this[_0x419e7d(0x19e)](),this[_0x419e7d(0x302)](),this[_0x419e7d(0x310)](),this[_0x419e7d(0x1d3)][_0x419e7d(0x3d7)](),this['createContents']();},Window_Message['prototype'][_0x22ff2b(0x233)]=function(_0xc8d8ea,_0x466c93){const _0x39ad03=_0x22ff2b;switch(_0xc8d8ea[_0x39ad03(0x1af)]()[_0x39ad03(0x415)]()){case _0x39ad03(0x364):this[_0x39ad03(0x25b)]=$gameActors[_0x39ad03(0x2c5)](_0x466c93);break;case _0x39ad03(0x2d9):this['_autoPositionTarget']=$gameParty[_0x39ad03(0x235)]()[_0x466c93-0x1];break;case _0x39ad03(0x2ca):this[_0x39ad03(0x25b)]=$gameTroop[_0x39ad03(0x235)]()[_0x466c93-0x1];break;case'map\x20player':this[_0x39ad03(0x25b)]=$gamePlayer;break;case'map\x20actor':const _0x7266c0=$gameActors[_0x39ad03(0x2c5)](_0x466c93)[_0x39ad03(0x1b9)]();if(_0x7266c0===0x0){if(_0x39ad03(0x362)!=='bADGB')return!![];else this['_autoPositionTarget']=$gamePlayer;}else{if(_0x39ad03(0x2de)!==_0x39ad03(0x2de))return _0x4f6ad7=_0x4a9305['replace'](/<B>/gi,_0x39ad03(0x2ad)),_0x56ccb6=_0x35b5c3[_0x39ad03(0x1d4)](/<\/B>/gi,_0x39ad03(0x30b)),_0x2c24a8=_0x107c48[_0x39ad03(0x1d4)](/<I>/gi,_0x39ad03(0x2ff)),_0x15d8ff=_0x5c2ad2[_0x39ad03(0x1d4)](/<\/I>/gi,_0x39ad03(0x1f3)),_0x30f994;else this[_0x39ad03(0x25b)]=$gamePlayer[_0x39ad03(0x1e6)]()['follower'](_0x7266c0-0x1);}break;case _0x39ad03(0x2e0):_0x466c93===0x1?this['_autoPositionTarget']=$gamePlayer:this['_autoPositionTarget']=$gamePlayer[_0x39ad03(0x1e6)]()[_0x39ad03(0x194)](_0x466c93-0x2);break;case _0x39ad03(0x321):this[_0x39ad03(0x25b)]=$gameMap[_0x39ad03(0x3e9)](_0x466c93);break;}this[_0x39ad03(0x25b)]&&this[_0x39ad03(0x260)]();},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x3b3)]=Window_Message[_0x22ff2b(0x3d4)]['synchronizeNameBox'],Window_Message['prototype'][_0x22ff2b(0x1f7)]=function(){const _0x1dd5d0=_0x22ff2b;this[_0x1dd5d0(0x260)](),VisuMZ[_0x1dd5d0(0x3d5)][_0x1dd5d0(0x3b3)][_0x1dd5d0(0x282)](this);},Window_Message['prototype'][_0x22ff2b(0x260)]=function(){const _0x1723f0=_0x22ff2b;if(!this[_0x1723f0(0x25b)])return;const _0x330f7b=SceneManager[_0x1723f0(0x261)];if(!_0x330f7b)return;if(!_0x330f7b[_0x1723f0(0x3fd)])return;const _0x299033=_0x330f7b['_spriteset']['findTargetSprite'](this[_0x1723f0(0x25b)]);if(!_0x299033)return;let _0x344d42=_0x299033['x'];_0x344d42-=this[_0x1723f0(0x244)]/0x2,_0x344d42-=(Graphics[_0x1723f0(0x244)]-Graphics['boxWidth'])/0x2;let _0x58d069=_0x299033['y'];_0x58d069-=this[_0x1723f0(0x293)],_0x58d069-=(Graphics[_0x1723f0(0x293)]-Graphics[_0x1723f0(0x36f)])/0x2,_0x58d069-=_0x299033['height']+0x8,this['x']=Math[_0x1723f0(0x278)](_0x344d42),this['y']=Math['round'](_0x58d069),this['clampPlacementPosition'](!![],![]),this[_0x1723f0(0x391)]['updatePlacement']();},Window_Message['prototype'][_0x22ff2b(0x332)]=function(){const _0x1f6544=_0x22ff2b;this[_0x1f6544(0x2d0)]=![],this[_0x1f6544(0x25b)]=undefined,$gameSystem[_0x1f6544(0x35d)](),this[_0x1f6544(0x2f9)](),this[_0x1f6544(0x239)]=0x0;},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x304)]=function(_0x2d928e){const _0x339d66=_0x22ff2b;return Window_Base['prototype']['preConvertEscapeCharacters'][_0x339d66(0x282)](this,_0x2d928e);},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x2e4)]=function(_0x538606){const _0x26c37e=_0x22ff2b;return Window_Base[_0x26c37e(0x3d4)][_0x26c37e(0x2e4)]['call'](this,_0x538606);},Window_Message[_0x22ff2b(0x3d4)]['flushTextState']=function(_0x37d04f){const _0x419bb6=_0x22ff2b;this[_0x419bb6(0x21a)](_0x37d04f),Window_Base[_0x419bb6(0x3d4)]['flushTextState']['call'](this,_0x37d04f),this[_0x419bb6(0x283)](_0x37d04f);},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x21a)]=function(_0x354432){},Window_Message[_0x22ff2b(0x3d4)][_0x22ff2b(0x283)]=function(_0xa467d0){},Window_NameBox[_0x22ff2b(0x3d4)][_0x22ff2b(0x223)]=function(){return![];},Window_NameBox['prototype'][_0x22ff2b(0x3bc)]=function(){const _0x2e0628=_0x22ff2b;Window_Base[_0x2e0628(0x3d4)]['resetTextColor'][_0x2e0628(0x282)](this),this[_0x2e0628(0x300)](this[_0x2e0628(0x363)]());},Window_NameBox[_0x22ff2b(0x3d4)][_0x22ff2b(0x363)]=function(){const _0x14e081=_0x22ff2b,_0x343c39=VisuMZ[_0x14e081(0x3d5)][_0x14e081(0x236)][_0x14e081(0x2da)]['NameBoxWindowDefaultColor'];return ColorManager[_0x14e081(0x3db)](_0x343c39);},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x3f1)]=Window_NameBox['prototype'][_0x22ff2b(0x19e)],Window_NameBox[_0x22ff2b(0x3d4)][_0x22ff2b(0x19e)]=function(){const _0x11a5d=_0x22ff2b;VisuMZ['MessageCore'][_0x11a5d(0x3f1)][_0x11a5d(0x282)](this),this['updateRelativePosition'](),this[_0x11a5d(0x3e4)](),this[_0x11a5d(0x1f4)](),this['updateOverlappingY']();},Window_NameBox['prototype'][_0x22ff2b(0x304)]=function(_0x4840a4){const _0x4ffefb=_0x22ff2b;return _0x4840a4=_0x4840a4[_0x4ffefb(0x1d4)](/<LEFT>/gi,this['setRelativePosition'][_0x4ffefb(0x1ff)](this,0x0)),_0x4840a4=_0x4840a4[_0x4ffefb(0x1d4)](/<CENTER>/gi,this[_0x4ffefb(0x22b)][_0x4ffefb(0x1ff)](this,0x5)),_0x4840a4=_0x4840a4[_0x4ffefb(0x1d4)](/<RIGHT>/gi,this[_0x4ffefb(0x22b)][_0x4ffefb(0x1ff)](this,0xa)),_0x4840a4=_0x4840a4['replace'](/<POSITION:[ ](\d+)>/gi,(_0x162df7,_0x4fa8c9)=>this[_0x4ffefb(0x22b)](parseInt(_0x4fa8c9))),_0x4840a4=_0x4840a4[_0x4ffefb(0x1d4)](/<\/LEFT>/gi,''),_0x4840a4=_0x4840a4[_0x4ffefb(0x1d4)](/<\/CENTER>/gi,''),_0x4840a4=_0x4840a4[_0x4ffefb(0x1d4)](/<\/RIGHT>/gi,''),Window_Base[_0x4ffefb(0x3d4)][_0x4ffefb(0x304)][_0x4ffefb(0x282)](this,_0x4840a4);},Window_NameBox[_0x22ff2b(0x3d4)][_0x22ff2b(0x22b)]=function(_0x23e21d){return this['_relativePosition']=_0x23e21d,'';},Window_NameBox['prototype']['updateRelativePosition']=function(){const _0x12b4cd=_0x22ff2b;if($gameMessage[_0x12b4cd(0x3b8)]())return;this[_0x12b4cd(0x208)]=this[_0x12b4cd(0x208)]||0x0;const _0x4b9716=this['_messageWindow'],_0x53a0b1=Math[_0x12b4cd(0x2db)](_0x4b9716[_0x12b4cd(0x244)]*this['_relativePosition']/0xa);this['x']=_0x4b9716['x']+_0x53a0b1-Math[_0x12b4cd(0x2db)](this[_0x12b4cd(0x244)]/0x2),this['x']=this['x'][_0x12b4cd(0x3a7)](_0x4b9716['x'],_0x4b9716['x']+_0x4b9716[_0x12b4cd(0x244)]-this[_0x12b4cd(0x244)]);},Window_NameBox[_0x22ff2b(0x3d4)][_0x22ff2b(0x3e4)]=function(){const _0x355ab1=_0x22ff2b;if($gameMessage[_0x355ab1(0x3b8)]())return;this['_relativePosition']=this['_relativePosition']||0x0;const _0x181b49=VisuMZ[_0x355ab1(0x3d5)][_0x355ab1(0x236)][_0x355ab1(0x2da)][_0x355ab1(0x30a)],_0x59e72d=VisuMZ['MessageCore']['Settings'][_0x355ab1(0x2da)][_0x355ab1(0x3da)],_0x33ed12=(0x5-this['_relativePosition'])/0x5;this['x']+=Math[_0x355ab1(0x2db)](_0x181b49*_0x33ed12),this['y']+=_0x59e72d;},Window_NameBox[_0x22ff2b(0x3d4)][_0x22ff2b(0x3c9)]=function(){const _0x5e7c40=_0x22ff2b,_0x2e94d6=this[_0x5e7c40(0x2a5)],_0x463ced=_0x2e94d6['y'],_0xdf6717=VisuMZ[_0x5e7c40(0x3d5)][_0x5e7c40(0x236)][_0x5e7c40(0x2da)][_0x5e7c40(0x3da)];_0x463ced>this['y']&&_0x463ced<this['y']+this[_0x5e7c40(0x293)]-_0xdf6717&&('myQTs'===_0x5e7c40(0x317)?this['y']=_0x2e94d6['y']+_0x2e94d6['height']:(this['_textColorStack']=this[_0x5e7c40(0x2d2)]||[],this['contents'][_0x5e7c40(0x3db)]=this[_0x5e7c40(0x2d2)][_0x5e7c40(0x3bf)]()||_0x4cb891[_0x5e7c40(0x309)]()));},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x251)]=Window_NameBox[_0x22ff2b(0x3d4)][_0x22ff2b(0x2b1)],Window_NameBox[_0x22ff2b(0x3d4)][_0x22ff2b(0x2b1)]=function(){const _0x3e5fac=_0x22ff2b;this[_0x3e5fac(0x208)]=0x0,VisuMZ[_0x3e5fac(0x3d5)][_0x3e5fac(0x251)]['call'](this);},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x375)]=function(){return![];},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x223)]=function(){return!![];},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x2c6)]=function(){return $gameSystem['getChoiceListLineHeight']()+0x8;},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x240)]=function(){const _0x1e34ec=_0x22ff2b;return $gameSystem[_0x1e34ec(0x2a6)]();},Window_ChoiceList['prototype']['start']=function(){const _0x3c882d=_0x22ff2b;this['refresh'](),this['selectDefault'](),this[_0x3c882d(0x3d8)](),this['activate']();},Window_ChoiceList['prototype'][_0x22ff2b(0x2b1)]=function(){const _0x2d330c=_0x22ff2b;this[_0x2d330c(0x1c8)](),this[_0x2d330c(0x407)](),this[_0x2d330c(0x2a5)]&&(this[_0x2d330c(0x19e)](),this[_0x2d330c(0x398)]()),this[_0x2d330c(0x367)](),this[_0x2d330c(0x1ea)](),this['refreshDimmerBitmap'](),Window_Selectable[_0x2d330c(0x3d4)][_0x2d330c(0x2b1)][_0x2d330c(0x282)](this);},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x407)]=function(){const _0x4b31ba=_0x22ff2b,_0x19c110=$gameMessage[_0x4b31ba(0x29b)]();let _0x40bb19=0x0;for(let _0x54a45b of _0x19c110){if(_0x4b31ba(0x393)!=='pIfNz'){_0x54a45b=this['convertChoiceMacros'](_0x54a45b);if(this['isChoiceVisible'](_0x54a45b)){const _0x48b5c9=this['parseChoiceText'](_0x54a45b),_0x4ba3f8=this['isChoiceEnabled'](_0x54a45b);this[_0x4b31ba(0x37f)](_0x48b5c9,_0x4b31ba(0x3a8),_0x4ba3f8,_0x40bb19);}_0x40bb19++;}else this[_0x4b31ba(0x29f)]=_0x153c43[_0x4b31ba(0x224)](_0x2daafa);}},Window_ChoiceList[_0x22ff2b(0x3d4)]['convertChoiceMacros']=function(_0x3e5d0b){const _0x438648=_0x22ff2b;return Window_Base[_0x438648(0x3d4)][_0x438648(0x377)][_0x438648(0x282)](this,_0x3e5d0b);},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x1c6)]=function(_0x35101d){const _0x25645d=_0x22ff2b;if(_0x35101d[_0x25645d(0x1f2)](/<HIDE>/i))return![];if(_0x35101d[_0x25645d(0x1f2)](/<SHOW>/i))return!![];if(_0x35101d[_0x25645d(0x1f2)](/<SHOW[ ](?:SW|SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){const _0x102930=JSON[_0x25645d(0x38f)]('['+RegExp['$1']['match'](/\d+/g)+']');for(const _0x470e29 of _0x102930){if(_0x25645d(0x2b0)!==_0x25645d(0x2e5)){if(!$gameSwitches[_0x25645d(0x290)](_0x470e29))return![];}else{if(this[_0x25645d(0x1ad)]===_0x3c5a86)this['initMessageCore']();if(this[_0x25645d(0x1ad)][_0x25645d(0x2f3)]===_0x3da15c)this['initMessageCore']();this['_MessageCoreSettings']['choiceTextAlign']=_0x498e24[_0x25645d(0x1af)]();}}return!![];}if(_0x35101d[_0x25645d(0x1f2)](/<SHOW ALL[ ](?:SW|SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){const _0x30a92d=JSON[_0x25645d(0x38f)]('['+RegExp['$1'][_0x25645d(0x1f2)](/\d+/g)+']');for(const _0x333e1b of _0x30a92d){if(!$gameSwitches[_0x25645d(0x290)](_0x333e1b))return![];}return!![];}if(_0x35101d['match'](/<SHOW ANY[ ](?:SW|SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){const _0x3ca7b8=JSON[_0x25645d(0x38f)]('['+RegExp['$1'][_0x25645d(0x1f2)](/\d+/g)+']');for(const _0x2591bb of _0x3ca7b8){if($gameSwitches[_0x25645d(0x290)](_0x2591bb))return!![];}return![];}if(_0x35101d['match'](/<HIDE[ ](?:SW|SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){const _0x46832f=JSON['parse']('['+RegExp['$1'][_0x25645d(0x1f2)](/\d+/g)+']');for(const _0xf01fd9 of _0x46832f){if(_0x25645d(0x297)!==_0x25645d(0x297))_0x37c4e2=_0x500105[_0x25645d(0x2c5)]();else{if(!$gameSwitches[_0x25645d(0x290)](_0xf01fd9))return!![];}}return![];}if(_0x35101d['match'](/<HIDE ALL[ ](?:SW|SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){const _0x57f883=JSON['parse']('['+RegExp['$1'][_0x25645d(0x1f2)](/\d+/g)+']');for(const _0x56e8f4 of _0x57f883){if(!$gameSwitches['value'](_0x56e8f4))return!![];}return![];}if(_0x35101d[_0x25645d(0x1f2)](/<HIDE ANY[ ](?:SW|SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){const _0x3882af=JSON[_0x25645d(0x38f)]('['+RegExp['$1'][_0x25645d(0x1f2)](/\d+/g)+']');for(const _0x952345 of _0x3882af){if($gameSwitches[_0x25645d(0x290)](_0x952345))return![];}return!![];}return!![];},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x29e)]=function(_0x4ed576){const _0x382c66=_0x22ff2b;let _0x5100fd=_0x4ed576;return _0x5100fd=_0x5100fd[_0x382c66(0x1d4)](/<(?:BR|LINEBREAK)>/gi,'\x0a'),_0x5100fd=_0x5100fd['replace'](/<LINE\x1bWrapBreak[0]BREAK>/gi,'\x0a'),_0x5100fd;},Window_ChoiceList[_0x22ff2b(0x3d4)]['isChoiceEnabled']=function(_0x257bf0){const _0x479378=_0x22ff2b;if(_0x257bf0[_0x479378(0x1f2)](/<DISABLE>/i))return![];if(_0x257bf0[_0x479378(0x1f2)](/<ENABLE>/i))return!![];if(_0x257bf0['match'](/<ENABLE[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){const _0x104e74=JSON[_0x479378(0x38f)]('['+RegExp['$1']['match'](/\d+/g)+']');for(const _0x5ab223 of _0x104e74){if(!$gameSwitches[_0x479378(0x290)](_0x5ab223))return![];}return!![];}if(_0x257bf0[_0x479378(0x1f2)](/<ENABLE ALL[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){if(_0x479378(0x3c2)!==_0x479378(0x2c8)){const _0x1f7172=JSON[_0x479378(0x38f)]('['+RegExp['$1'][_0x479378(0x1f2)](/\d+/g)+']');for(const _0x108e24 of _0x1f7172){if('QRAqu'===_0x479378(0x31b))return(_0x472853[_0x479378(0x26f)]-this[_0x479378(0x1aa)]())/0x2;else{if(!$gameSwitches[_0x479378(0x290)](_0x108e24))return![];}}return!![];}else{const _0x296c05=_0x553288['MessageCore'][_0x479378(0x236)][_0x479378(0x2da)],_0x504cc3=_0x5e8c02[_0x479378(0x3d5)][_0x479378(0x236)][_0x479378(0x416)];this[_0x479378(0x1ad)]={'messageRows':_0x296c05['MessageRows'],'messageWidth':_0x296c05['MessageWidth'],'messageWordWrap':_0x504cc3[_0x479378(0x227)],'helpWordWrap':_0x504cc3['HelpWindow'],'choiceLineHeight':_0x296c05[_0x479378(0x20a)],'choiceRows':_0x296c05['ChoiceWindowMaxRows'],'choiceCols':_0x296c05[_0x479378(0x1b7)],'choiceTextAlign':_0x296c05['ChoiceWindowTextAlign']};}}if(_0x257bf0[_0x479378(0x1f2)](/<ENABLE ANY[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){const _0x33d040=JSON[_0x479378(0x38f)]('['+RegExp['$1'][_0x479378(0x1f2)](/\d+/g)+']');for(const _0x8d48b4 of _0x33d040){if($gameSwitches[_0x479378(0x290)](_0x8d48b4))return!![];}return![];}if(_0x257bf0[_0x479378(0x1f2)](/<DISABLE[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){if(_0x479378(0x2e6)===_0x479378(0x2e6)){const _0x20f260=JSON['parse']('['+RegExp['$1'][_0x479378(0x1f2)](/\d+/g)+']');for(const _0xae0b62 of _0x20f260){if(_0x479378(0x32f)!==_0x479378(0x25d)){if(!$gameSwitches[_0x479378(0x290)](_0xae0b62))return!![];}else return this[_0x479378(0x2a5)]?this[_0x479378(0x1c3)]():_0xf3461d[_0x479378(0x3d5)][_0x479378(0x20d)]['call'](this);}return![];}else _0x4251f5['update']();}if(_0x257bf0[_0x479378(0x1f2)](/<DISABLE ALL[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){if(_0x479378(0x3ac)!==_0x479378(0x3ac))this['moveTo'](this[_0x479378(0x29f)]['x'],this[_0x479378(0x387)]*(_0x3e9162[_0x479378(0x36f)]-this['height'])/0x2,this[_0x479378(0x29f)][_0x479378(0x244)],this[_0x479378(0x29f)][_0x479378(0x293)],_0xb3c126,_0x43db10);else{const _0x546d10=JSON[_0x479378(0x38f)]('['+RegExp['$1'][_0x479378(0x1f2)](/\d+/g)+']');for(const _0x518312 of _0x546d10){if(!$gameSwitches[_0x479378(0x290)](_0x518312))return!![];}return![];}}if(_0x257bf0[_0x479378(0x1f2)](/<DISABLE ANY[ ](?:SWITCH|SWITCHES):[ ]*(\d+(?:\s*,\s*\d+)*)>/i)){if(_0x479378(0x3b2)!==_0x479378(0x3c7)){const _0x15afb2=JSON['parse']('['+RegExp['$1'][_0x479378(0x1f2)](/\d+/g)+']');for(const _0xf269ff of _0x15afb2){if(_0x479378(0x3e1)===_0x479378(0x41c))_0x83c16a['x']=this['obtainEscapeParam'](_0x35d847),_0x1102d3[_0x479378(0x3d5)][_0x479378(0x236)][_0x479378(0x2da)][_0x479378(0x222)]&&(_0x15666b['x']+=_0x1e404a[_0x479378(0x3ee)]);else{if($gameSwitches[_0x479378(0x290)](_0xf269ff))return![];}}return!![];}else return this['processAutoSize'](_0x2e2cb0,!![],![]),this[_0x479378(0x233)](_0x479378(0x266)),'';}return!![];},VisuMZ[_0x22ff2b(0x3d5)][_0x22ff2b(0x1a9)]=Window_ChoiceList[_0x22ff2b(0x3d4)]['updatePlacement'],Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x19e)]=function(){const _0x807854=_0x22ff2b;VisuMZ[_0x807854(0x3d5)][_0x807854(0x1a9)][_0x807854(0x282)](this),this[_0x807854(0x1f4)]();},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x398)]=function(){const _0x409d65=_0x22ff2b;if(!this[_0x409d65(0x36a)])return;const _0x5ab149=0x8,_0x26d7a3=this[_0x409d65(0x36a)],_0xe30efa=this['x']+this[_0x409d65(0x244)],_0x4063f8=Math[_0x409d65(0x2db)]((Graphics[_0x409d65(0x244)]-Graphics[_0x409d65(0x26f)])/0x2);_0xe30efa>=Graphics[_0x409d65(0x26f)]+_0x4063f8-_0x26d7a3[_0x409d65(0x244)]+_0x5ab149?_0x26d7a3['x']=-_0x26d7a3[_0x409d65(0x244)]-_0x5ab149:_0x26d7a3['x']=this['width']+_0x5ab149,_0x26d7a3['y']=this[_0x409d65(0x293)]/0x2-_0x26d7a3['height']/0x2;},VisuMZ[_0x22ff2b(0x3d5)]['Window_ChoiceList_windowX']=Window_ChoiceList[_0x22ff2b(0x3d4)]['windowX'],Window_ChoiceList[_0x22ff2b(0x3d4)]['windowX']=function(){const _0xffb066=_0x22ff2b;return this[_0xffb066(0x2a5)]?this[_0xffb066(0x1c3)]():VisuMZ[_0xffb066(0x3d5)][_0xffb066(0x20d)][_0xffb066(0x282)](this);},Window_ChoiceList['prototype']['messageCoreWindowX']=function(){const _0x42e883=_0x22ff2b,_0x34a2e9=$gameMessage[_0x42e883(0x23f)]();if(_0x34a2e9===0x1){if(_0x42e883(0x353)===_0x42e883(0x1d5))_0x3acbb5-=_0x2e1d83;else return(Graphics[_0x42e883(0x26f)]-this[_0x42e883(0x1aa)]())/0x2;}else return _0x34a2e9===0x2?this[_0x42e883(0x2a5)]['x']+this[_0x42e883(0x2a5)][_0x42e883(0x244)]-this[_0x42e883(0x1aa)]():this[_0x42e883(0x2a5)]['x'];},Window_ChoiceList['prototype'][_0x22ff2b(0x1aa)]=function(){const _0x27b127=_0x22ff2b,_0x2b4fcf=(this[_0x27b127(0x357)]()+this['colSpacing']())*this['maxCols']()+this[_0x27b127(0x26c)]*0x2;return Math[_0x27b127(0x2cb)](_0x2b4fcf,Graphics[_0x27b127(0x244)]);},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x1e8)]=function(){const _0x3bae96=_0x22ff2b,_0x3a6e62=$gameMessage[_0x3bae96(0x29b)]()[_0x3bae96(0x384)](_0x192f5b=>this[_0x3bae96(0x28e)](_0x192f5b))[_0x3bae96(0x397)](_0x3c6d25=>this[_0x3bae96(0x1c6)](_0x3c6d25)),_0x40c17c=Math[_0x3bae96(0x250)](_0x3a6e62['length']/this[_0x3bae96(0x240)]());return Math['max'](0x1,Math[_0x3bae96(0x2cb)](_0x40c17c,this['maxLines']()));},Window_ChoiceList[_0x22ff2b(0x3d4)]['maxLines']=function(){const _0x53ac8d=_0x22ff2b,_0x577af4=this[_0x53ac8d(0x2a5)],_0x3ae919=_0x577af4?_0x577af4['y']:0x0,_0x47da2e=_0x577af4?_0x577af4[_0x53ac8d(0x293)]:0x0,_0x3cc681=Graphics[_0x53ac8d(0x36f)]/0x2;return _0x3ae919<_0x3cc681&&_0x3ae919+_0x47da2e>_0x3cc681?0x4:_0x53ac8d(0x2e2)===_0x53ac8d(0x2e2)?$gameSystem['getChoiceListMaxRows']():_0x4ab02a[_0x53ac8d(0x1de)]();},Window_ChoiceList['prototype'][_0x22ff2b(0x357)]=function(){const _0x19d2d1=_0x22ff2b;let _0xef46fe=0x60;for(const _0x3b9974 of this[_0x19d2d1(0x347)]){const _0x42ae5f=_0x3b9974[_0x19d2d1(0x36d)],_0xc9b99b=this[_0x19d2d1(0x37d)](_0x42ae5f)[_0x19d2d1(0x244)],_0x4109db=Math[_0x19d2d1(0x250)](_0xc9b99b)+this[_0x19d2d1(0x37c)]()*0x2;if(_0xef46fe<_0x4109db){if(_0x19d2d1(0x3fb)===_0x19d2d1(0x286)){const _0x59ec9a=_0x876c20[_0x19d2d1(0x38f)]('['+_0x166db9['$1'][_0x19d2d1(0x1f2)](/\d+/g)+']');for(const _0x4d1e85 of _0x59ec9a){if(!_0x4f5cdb['value'](_0x4d1e85))return!![];}return![];}else _0xef46fe=_0x4109db;}}return _0xef46fe;},Window_ChoiceList[_0x22ff2b(0x3d4)]['drawItem']=function(_0x29cbe6){const _0x1b1802=_0x22ff2b,_0x45fb73=this['itemRectWithPadding'](_0x29cbe6),_0x238eb0=$gameSystem[_0x1b1802(0x1ae)]()!==_0x1b1802(0x1d1)?_0x1b1802(0x3f9)[_0x1b1802(0x197)]($gameSystem[_0x1b1802(0x1ae)]()):'',_0x4b3b87=_0x238eb0+this[_0x1b1802(0x24e)](_0x29cbe6);this['changePaintOpacity'](this[_0x1b1802(0x3aa)](_0x29cbe6));const _0x1a71b7=this[_0x1b1802(0x37d)](_0x4b3b87)[_0x1b1802(0x293)],_0x536f26=Math[_0x1b1802(0x2d3)](_0x45fb73['y'],_0x45fb73['y']+Math[_0x1b1802(0x278)]((_0x45fb73[_0x1b1802(0x293)]-_0x1a71b7)/0x2));this[_0x1b1802(0x2f1)](_0x4b3b87,_0x45fb73['x'],_0x536f26,_0x45fb73[_0x1b1802(0x244)]);},Window_ChoiceList[_0x22ff2b(0x3d4)][_0x22ff2b(0x1b3)]=function(){const _0x103709=_0x22ff2b;$gameMessage[_0x103709(0x25e)](this[_0x103709(0x1e1)]()),this[_0x103709(0x2a5)][_0x103709(0x3c6)](),this[_0x103709(0x1d2)]();};